<?php

namespace Drupal\ai_upgrade_assistant\Theme;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Template\Attribute;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides preprocessing for AI Upgrade Assistant theme elements.
 */
class AiUpgradeAssistantPreprocess implements ContainerInjectionInterface {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new AiUpgradeAssistantPreprocess object.
   */
  public function __construct(ConfigFactoryInterface $configFactory) {
    $this->configFactory = $configFactory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory')
    );
  }

  /**
   * Preprocesses the upgrade dashboard.
   */
  public function preprocessDashboard(&$variables) {
    $config = $this->configFactory->get('ai_upgrade_assistant.settings');
    
    // Add theme settings
    $variables['dashboard_theme'] = $config->get('dashboard_theme') ?? 'modern';
    $variables['show_animations'] = $config->get('show_animations') ?? TRUE;
    $variables['compact_view'] = $config->get('compact_view') ?? FALSE;
    
    // Add dashboard attributes
    $variables['dashboard_attributes'] = new Attribute([
      'class' => [
        'ai-upgrade-dashboard',
        'theme-' . $variables['dashboard_theme'],
        $variables['show_animations'] ? 'with-animations' : 'no-animations',
        $variables['compact_view'] ? 'compact-view' : 'full-view',
      ],
    ]);
    
    // Add dynamic sections based on user permissions
    $variables['sections'] = $this->getDashboardSections();
    
    // Add custom CSS variables for theming
    $variables['#attached']['drupalSettings']['aiUpgradeAssistant']['theme'] = [
      'primaryColor' => $config->get('primary_color') ?? '#0078d4',
      'secondaryColor' => $config->get('secondary_color') ?? '#2ecc71',
      'fontFamily' => $config->get('font_family') ?? 'system-ui, -apple-system, sans-serif',
      'borderRadius' => $config->get('border_radius') ?? '4px',
    ];
  }

  /**
   * Preprocesses the upgrade analysis results.
   */
  public function preprocessAnalysisResults(&$variables) {
    $config = $this->configFactory->get('ai_upgrade_assistant.settings');
    
    // Add result display settings
    $variables['show_code_previews'] = $config->get('show_code_previews') ?? TRUE;
    $variables['group_by_severity'] = $config->get('group_by_severity') ?? TRUE;
    $variables['inline_diffs'] = $config->get('inline_diffs') ?? TRUE;
    
    // Add custom styling for different severity levels
    $variables['severity_colors'] = [
      'critical' => $config->get('critical_color') ?? '#e74c3c',
      'warning' => $config->get('warning_color') ?? '#f1c40f',
      'info' => $config->get('info_color') ?? '#3498db',
      'success' => $config->get('success_color') ?? '#2ecc71',
    ];

    // Process module information
    if (!empty($variables['analysis']['modules'])) {
      foreach ($variables['analysis']['modules'] as &$module) {
        // Add type class
        $module_info = \Drupal::service('extension.list.module')->get($module['name']);
        $module['type'] = $module_info->origin === 'core' ? 'core' : ($module_info->origin === 'profile' ? 'profile' : 'contrib');
        
        // Format version information
        if (!empty($module['version'])) {
          $module['version_formatted'] = t('Version @version', ['@version' => $module['version']]);
        }
        else {
          $module['version_formatted'] = t('Version unknown');
        }

        // Add status class
        $module['status_class'] = strtolower($module['status']);

        // Format compatibility score
        if (isset($module['compatibility'])) {
          $module['compatibility_class'] = $this->getCompatibilityClass($module['compatibility']);
          $module['compatibility_formatted'] = $this->formatCompatibilityScore($module['compatibility']);
        }

        // Process issues
        if (!empty($module['issues'])) {
          foreach ($module['issues'] as &$issue) {
            $issue['severity_class'] = strtolower($issue['severity']);
            $issue['severity_icon'] = $this->getSeverityIcon($issue['severity']);
          }
        }
      }
    }
  }

  /**
   * Gets the compatibility class based on score.
   *
   * @param int $score
   *   The compatibility score.
   *
   * @return string
   *   The CSS class for the compatibility level.
   */
  protected function getCompatibilityClass($score) {
    if ($score >= 90) {
      return 'high-compatibility';
    }
    elseif ($score >= 70) {
      return 'medium-compatibility';
    }
    else {
      return 'low-compatibility';
    }
  }

  /**
   * Formats the compatibility score.
   *
   * @param int $score
   *   The compatibility score.
   *
   * @return string
   *   The formatted score string.
   */
  protected function formatCompatibilityScore($score) {
    return t('@score% compatible', ['@score' => $score]);
  }

  /**
   * Gets the icon for a severity level.
   *
   * @param string $severity
   *   The severity level.
   *
   * @return string
   *   The material icon name.
   */
  protected function getSeverityIcon($severity) {
    switch ($severity) {
      case 'critical':
        return 'error';
      case 'error':
        return 'warning';
      case 'warning':
        return 'info';
      default:
        return 'check_circle';
    }
  }

  /**
   * Gets dashboard sections based on user permissions.
   *
   * @return array
   *   Array of dashboard sections with their settings.
   */
  protected function getDashboardSections() {
    $sections = [];
    
    // Module Overview Section
    $sections['overview'] = [
      'title' => t('Module Overview'),
      'weight' => 0,
      'collapsed' => FALSE,
    ];
    
    // Upgrade Analysis Section
    if (\Drupal::currentUser()->hasPermission('run upgrade analysis')) {
      $sections['analysis'] = [
        'title' => t('Upgrade Analysis'),
        'weight' => 10,
        'collapsed' => FALSE,
      ];
    }
    
    // Pattern Learning Section
    if (\Drupal::currentUser()->hasPermission('manage upgrade patterns')) {
      $sections['patterns'] = [
        'title' => t('Pattern Learning'),
        'weight' => 20,
        'collapsed' => TRUE,
      ];
    }
    
    // Community Insights Section
    if (\Drupal::currentUser()->hasPermission('view community insights')) {
      $sections['community'] = [
        'title' => t('Community Insights'),
        'weight' => 30,
        'collapsed' => TRUE,
      ];
    }
    
    // Sort sections by weight
    uasort($sections, function ($a, $b) {
      return $a['weight'] <=> $b['weight'];
    });
    
    return $sections;
  }

}
