<?php

namespace Drupal\Tests\ai_upgrade_assistant\Unit\Service\BatchAnalyzer;

use Drupal\ai_upgrade_assistant\Service\BatchAnalyzer;
use Drupal\ai_upgrade_assistant\Service\ProjectAnalyzer;
use Drupal\ai_upgrade_assistant\Service\AnalysisTracker;
use Drupal\ai_upgrade_assistant\Service\PatchSearcher;
use Drupal\Tests\UnitTestCase;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Cache\CacheBackendInterface;

/**
 * Tests the batch analyzer service.
 *
 * @group ai_upgrade_assistant
 * @coversDefaultClass \Drupal\ai_upgrade_assistant\Service\BatchAnalyzer
 */
class BatchAnalyzerTest extends UnitTestCase {

  /**
   * The batch analyzer service under test.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\BatchAnalyzer
   */
  protected $batchAnalyzer;

  /**
   * The mock project analyzer.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\ProjectAnalyzer
   */
  protected $projectAnalyzer;

  /**
   * The mock state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The mock module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The mock analysis tracker.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\AnalysisTracker
   */
  protected $analysisTracker;

  /**
   * The mock patch searcher.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\PatchSearcher
   */
  protected $patchSearcher;

  /**
   * The mock cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->projectAnalyzer = $this->createMock(ProjectAnalyzer::class);
    $this->state = $this->createMock(StateInterface::class);
    $this->moduleHandler = $this->createMock(ModuleHandlerInterface::class);
    $this->analysisTracker = $this->createMock(AnalysisTracker::class);
    $this->patchSearcher = $this->createMock(PatchSearcher::class);
    $this->cache = $this->createMock(CacheBackendInterface::class);

    $this->batchAnalyzer = new BatchAnalyzer(
      $this->projectAnalyzer,
      $this->state,
      $this->moduleHandler,
      $this->analysisTracker,
      $this->patchSearcher,
      $this->cache
    );
  }

  /**
   * Tests batch operation creation.
   *
   * @covers ::createBatchOperations
   */
  public function testBatchOperationCreation() {
    $modules = [
      'test_module1' => (object) ['getPath' => '/path/to/module1'],
      'test_module2' => (object) ['getPath' => '/path/to/module2'],
    ];

    $this->moduleHandler->expects($this->once())
      ->method('getModuleList')
      ->willReturn($modules);

    $operations = $this->batchAnalyzer->createBatchOperations();
    
    $this->assertIsArray($operations);
    $this->assertCount(2, $operations);
    $this->assertEquals('analyzeSingleModule', $operations[0][0][1]);
  }

  /**
   * Tests batch processing.
   *
   * @covers ::processBatch
   */
  public function testBatchProcessing() {
    $context = ['results' => [], 'finished' => 0];
    $module_name = 'test_module';
    $module_path = '/path/to/module';

    $this->projectAnalyzer->expects($this->once())
      ->method('analyzeModule')
      ->with($module_name, $module_path)
      ->willReturn(['status' => 'success']);

    $this->batchAnalyzer->processBatch($module_name, $module_path, [], $context);
    
    $this->assertArrayHasKey($module_name, $context['results']);
    $this->assertEquals(1, $context['finished']);
  }

  /**
   * Tests caching system.
   *
   * @covers ::getCachedAnalysis
   */
  public function testCaching() {
    $module_name = 'test_module';
    $cached_data = ['status' => 'cached'];

    $this->cache->expects($this->once())
      ->method('get')
      ->with('ai_upgrade_assistant.analysis.' . $module_name)
      ->willReturn((object) ['data' => $cached_data]);

    $result = $this->batchAnalyzer->getCachedAnalysis($module_name);
    
    $this->assertEquals($cached_data, $result);
  }

  /**
   * Tests progress tracking.
   *
   * @covers ::trackProgress
   */
  public function testProgressTracking() {
    $module_name = 'test_module';
    $status = 'in_progress';

    $this->analysisTracker->expects($this->once())
      ->method('updateStatus')
      ->with($module_name, $status);

    $this->batchAnalyzer->trackProgress($module_name, $status);
  }

  /**
   * Tests resource optimization.
   *
   * @covers ::optimizeResources
   */
  public function testResourceOptimization() {
    $memory_limit = '256M';
    $time_limit = 300;

    $result = $this->batchAnalyzer->optimizeResources($memory_limit, $time_limit);
    
    $this->assertTrue($result);
    $this->assertEquals($memory_limit, ini_get('memory_limit'));
    $this->assertEquals($time_limit, ini_get('max_execution_time'));
  }
}
