<?php

namespace Drupal\ai_upgrade_assistant\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\Process\Process;
use Symfony\Component\Process\Exception\ProcessFailedException;
use Drupal\ai_upgrade_assistant\Service\ProjectAnalyzer;
use Drupal\ai_upgrade_assistant\Service\PatchSearcher;
use Drupal\ai_upgrade_assistant\Service\BatchAnalyzer;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Extension\ModuleExtensionList;

/**
 * Controller for the upgrade process.
 *
 * @package Drupal\ai_upgrade_assistant\Controller
 */
class UpgradeController extends ControllerBase {

  /**
   * The project analyzer service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\ProjectAnalyzer
   */
  protected $projectAnalyzer;

  /**
   * The patch searcher service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\PatchSearcher
   */
  protected $patchSearcher;

  /**
   * The batch analyzer service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\BatchAnalyzer
   */
  protected $batchAnalyzer;

  /**
   * The module extension list service.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected $moduleExtensionList;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * Constructs a new UpgradeController object.
   *
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   * @param \Drupal\Core\Extension\ModuleExtensionList $module_extension_list
   *   The module extension list service.
   * @param \Drupal\ai_upgrade_assistant\Service\ProjectAnalyzer $project_analyzer
   *   The project analyzer service.
   * @param \Drupal\ai_upgrade_assistant\Service\PatchSearcher $patch_searcher
   *   The patch searcher service.
   * @param \Drupal\ai_upgrade_assistant\Service\BatchAnalyzer $batch_analyzer
   *   The batch analyzer service.
   */
  public function __construct(
    StateInterface $state,
    ModuleHandlerInterface $module_handler,
    ModuleExtensionList $module_extension_list,
    ProjectAnalyzer $project_analyzer,
    PatchSearcher $patch_searcher,
    BatchAnalyzer $batch_analyzer
  ) {
    $this->state = $state;
    $this->moduleHandler = $module_handler;
    $this->moduleExtensionList = $module_extension_list;
    $this->projectAnalyzer = $project_analyzer;
    $this->patchSearcher = $patch_searcher;
    $this->batchAnalyzer = $batch_analyzer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('state'),
      $container->get('module_handler'),
      $container->get('extension.list.module'),
      $container->get('ai_upgrade_assistant.project_analyzer'),
      $container->get('ai_upgrade_assistant.patch_searcher'),
      $container->get('ai_upgrade_assistant.batch_analyzer')
    );
  }

  /**
   * Check environment requirements.
   */
  protected function checkEnvironment() {
    $requirements = [];
    
    // Check PHP version
    $current_php = PHP_VERSION;
    $minimum_php = '8.1.0';  // Current minimum requirement
    $recommended_php = '8.3.0';  // Recommended for future compatibility
    
    $minimum_status = version_compare($current_php, $minimum_php, '>=');
    $recommended_status = version_compare($current_php, $recommended_php, '>=');
    
    $requirements[] = [
      'title' => $this->t('PHP Version'),
      'value' => $current_php,
      'status' => $minimum_status, // Only fail if below absolute minimum
      'command' => $minimum_status 
        ? ($recommended_status ? null : $this->t('Suggestion: Consider upgrading PHP to @version for better future compatibility.', [
            '@version' => $recommended_php,
          ]))
        : $this->t('Warning: PHP @current is below the minimum requirement of @minimum.', [
            '@current' => $current_php,
            '@minimum' => $minimum_php,
          ]),
    ];

    // Check Composer version
    $composer_version = $this->getComposerVersion();
    $required_composer = '2.6.0';
    $composer_status = version_compare($composer_version, $required_composer, '>=');
    
    $requirements[] = [
      'title' => $this->t('Composer Version'),
      'value' => $composer_version,
      'status' => $composer_status,
      'command' => $composer_status ? null : 'composer self-update',
    ];

    // Check memory limit
    $memory_limit = ini_get('memory_limit');
    $required_memory = '1024M';
    $memory_status = $this->convertToBytes($memory_limit) >= $this->convertToBytes($required_memory);
    
    $requirements[] = [
      'title' => $this->t('Memory Limit'),
      'value' => $memory_limit,
      'status' => $memory_status,
      'command' => $memory_status ? null : 'php -d memory_limit=1024M',
    ];

    // Add terminal message about environment status
    foreach ($requirements as $requirement) {
      if (!$requirement['status']) {
        // Critical requirements not met
        $this->addTerminalMessage(
          $this->t('Warning: @title (@value) is below minimum requirement. @command', [
            '@title' => $requirement['title'],
            '@value' => $requirement['value'],
            '@command' => $requirement['command'],
          ]),
          'warning'
        );
      } elseif (!empty($requirement['command'])) {
        // Suggestions for improvement
        $this->addTerminalMessage(
          $this->t('Note: @title (@value) - @command', [
            '@title' => $requirement['title'],
            '@value' => $requirement['value'],
            '@command' => $requirement['command'],
          ]),
          'info'
        );
      }
    }

    if (empty(array_filter($requirements, fn($req) => !$req['status']))) {
      $this->addTerminalMessage(
        $this->t('All minimum requirements are met.'),
        'success'
      );
    }

    return $requirements;
  }

  /**
   * Get modules grouped by status.
   *
   * @return array
   *   An array of modules grouped by status.
   */
  protected function getModulesByStatus() {
    $modules = [
      'compatible' => [],
      'needs_update' => [],
      'unknown' => [],
    ];

    $installed_modules = $this->moduleHandler->getModuleList();
    
    foreach ($installed_modules as $name => $extension) {
      if ($this->isModuleCompatible($name)) {
        $modules['compatible'][] = [
          'name' => $name,
          'info' => $this->moduleExtensionList->getExtensionInfo($name),
          'has_fix' => $this->hasKnownFix($name),
        ];
      }
      else {
        $modules['needs_update'][] = [
          'name' => $name,
          'info' => $this->moduleExtensionList->getExtensionInfo($name),
          'has_fix' => $this->hasKnownFix($name),
        ];
      }
    }

    return $modules;
  }

  /**
   * Calculate overall progress.
   *
   * @return array
   *   Progress information.
   */
  protected function calculateProgress() {
    $modules = $this->getModulesByStatus();
    $total = count($modules['compatible']) + count($modules['needs_update']);
    $completed = count($modules['compatible']);

    return [
      'total' => $total,
      'completed' => $completed,
      'percentage' => $total > 0 ? round(($completed / $total) * 100) : 0,
    ];
  }

  /**
   * Automatically fixes all modules that can be automatically upgraded.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response object.
   */
  public function autoFixAll() {
    $modules = $this->getModulesByStatus();
    $batch = [
      'title' => $this->t('Upgrading modules...'),
      'operations' => [],
      'finished' => [$this, 'batchFinished'],
      'progress_message' => $this->t('Processed @current out of @total modules.'),
      'error_message' => $this->t('An error occurred during the upgrade process.'),
    ];

    foreach ($modules['needs_update'] as $module) {
      if ($module['has_fix']) {
        $batch['operations'][] = [
          [$this, 'batchProcessModule'],
          [$module],
        ];
      }
    }

    batch_set($batch);
    return $this->redirect('ai_upgrade_assistant.status');
  }

  /**
   * Auto-fixes a single module.
   *
   * @param string $module_name
   *   The name of the module to fix.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response to the status page.
   */
  public function autoFixModule($module_name) {
    $modules = $this->getModulesByStatus();
    $module = null;

    foreach ($modules['needs_update'] as $mod) {
      if ($mod['name'] === $module_name) {
        $module = $mod;
        break;
      }
    }

    if ($module && $module['has_fix']) {
      $batch = [
        'title' => $this->t('Upgrading @module...', ['@module' => $module_name]),
        'operations' => [
          [
            [$this, 'batchProcessModule'],
            [$module],
          ],
        ],
        'finished' => [$this, 'batchFinished'],
        'progress_message' => $this->t('Processing @module...', ['@module' => $module_name]),
        'error_message' => $this->t('An error occurred during the upgrade process.'),
      ];

      batch_set($batch);
    }

    return $this->redirect('ai_upgrade_assistant.status');
  }

  /**
   * Process a single module in the batch operation.
   *
   * @param array $module
   *   The module to process.
   * @param array &$context
   *   The batch context array.
   */
  public function batchProcessModule($module, &$context) {
    if (!isset($context['sandbox']['progress'])) {
      $context['sandbox']['progress'] = 0;
      $context['sandbox']['current_module'] = 0;
      $context['sandbox']['max'] = 1;
    }

    try {
      // Analyze module
      $analysis = $this->projectAnalyzer->analyzeModule($module['name']);
      
      // Search for patches
      $patches = $this->patchSearcher->findPatches($module['name'], $analysis);
      
      // Process module with batch analyzer
      $this->batchAnalyzer->processModule($module['name'], $analysis, $patches);

      $context['message'] = $this->t('Upgraded @name module.', ['@name' => $module['name']]);
      $context['results'][] = $module['name'];
    }
    catch (\Exception $e) {
      $context['results']['errors'][] = $this->t('Error upgrading @name: @error', [
        '@name' => $module['name'],
        '@error' => $e->getMessage(),
      ]);
    }

    $context['sandbox']['progress']++;
    $context['sandbox']['current_module']++;
    $context['finished'] = $context['sandbox']['progress'] / $context['sandbox']['max'];
  }

  /**
   * Finish batch processing.
   *
   * @param bool $success
   *   Whether the batch completed successfully.
   * @param array $results
   *   The batch results array.
   * @param array $operations
   *   The batch operations array.
   */
  public function batchFinished($success, $results, $operations) {
    if ($success) {
      if (!empty($results['errors'])) {
        foreach ($results['errors'] as $error) {
          $this->messenger()->addError($error);
        }
      }
      else {
        $count = count($results);
        $this->messenger()->addStatus($this->formatPlural(
          $count,
          'Successfully upgraded 1 module.',
          'Successfully upgraded @count modules.'
        ));
      }
    }
    else {
      $this->messenger()->addError($this->t('An error occurred while upgrading the modules.'));
    }
  }

  /**
   * Displays the upgrade status page.
   *
   * @return array
   *   A render array for the status page.
   */
  public function status() {
    $modules = $this->getModulesByStatus();
    $progress = $this->calculateProgress();
    $environment = $this->checkEnvironment();

    return [
      '#theme' => 'ai_upgrade_assistant_status',
      '#modules' => $modules,
      '#progress' => $progress,
      '#environment' => $environment,
      '#attached' => [
        'library' => ['ai_upgrade_assistant/status_page'],
        'drupalSettings' => [
          'aiUpgradeAssistant' => [
            'statusUrl' => Url::fromRoute('ai_upgrade_assistant.check_status')->toString(),
          ],
        ],
      ],
    ];
  }

  /**
   * Checks the current upgrade status.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response containing the current upgrade status.
   */
  public function checkStatus() {
    $response = [
      'status' => 'in_progress',
      'message' => '',
      'progress' => 0,
      'errors' => [],
      'terminal_output' => [],
    ];

    try {
      // Get stored state
      $state = $this->state->get('ai_upgrade_assistant.upgrade_status', []);
      if (!empty($state)) {
        $response = array_merge($response, $state);
      }

      // Calculate current progress
      $progress = $this->calculateProgress();
      $response['progress'] = $progress['percentage'];

      // Get modules by status
      $modules = $this->getModulesByStatus();
      $response['modules'] = [
        'compatible' => count($modules['compatible']),
        'needs_update' => count($modules['needs_update']),
      ];

      // Check if process is complete
      if ($progress['percentage'] === 100) {
        $response['status'] = 'complete';
        $response['message'] = $this->t('Upgrade process completed successfully.');
      }

      // Get terminal output
      $terminal_output = $this->state->get('ai_upgrade_assistant.terminal_output', []);
      if (!empty($terminal_output)) {
        $response['terminal_output'] = $terminal_output;
      }
    }
    catch (\Exception $e) {
      $response['status'] = 'error';
      $response['message'] = $this->t('Error checking upgrade status: @error', ['@error' => $e->getMessage()]);
      $response['errors'][] = $e->getMessage();
    }

    return new JsonResponse($response);
  }

  /**
   * Initialize the terminal output with default messages.
   */
  protected function initializeTerminalOutput() {
    $this->state->set('ai_upgrade_assistant.terminal_output', [
      [
        'message' => $this->t('Initializing AI Upgrade Assistant...'),
        'type' => 'info',
        'timestamp' => time(),
      ],
    ]);
  }

  /**
   * Add a message to the terminal output.
   *
   * @param string $message
   *   The message to add.
   * @param string $type
   *   The message type (info, success, warning, error).
   *
   * @return array
   *   The updated terminal output array.
   */
  protected function addTerminalMessage($message, $type = 'info') {
    $terminal_output = $this->state->get('ai_upgrade_assistant.terminal_output', []);
    
    $terminal_output[] = [
      'message' => $message,
      'type' => $type,
      'timestamp' => time(),
    ];
    
    $this->state->set('ai_upgrade_assistant.terminal_output', $terminal_output);
    
    return $terminal_output;
  }

  /**
   * Get the installed Composer version.
   *
   * @return string
   *   The Composer version string.
   */
  protected function getComposerVersion() {
    try {
      $process = new Process(['composer', '--version']);
      $process->run();
      
      if (!$process->isSuccessful()) {
        throw new ProcessFailedException($process);
      }
      
      $output = $process->getOutput();
      preg_match('/Composer version ([0-9.]+)/', $output, $matches);
      
      return $matches[1] ?? '0.0.0';
    }
    catch (\Exception $e) {
      return '0.0.0';
    }
  }

  /**
   * Convert PHP memory limit string to bytes.
   *
   * @param string $memory_limit
   *   Memory limit string (e.g., '128M', '1G').
   *
   * @return int
   *   Memory limit in bytes.
   */
  protected function convertToBytes($memory_limit) {
    $value = (int) $memory_limit;
    
    switch (strtolower(substr($memory_limit, -1))) {
      case 'g':
        $value *= 1024;
      case 'm':
        $value *= 1024;
      case 'k':
        $value *= 1024;
    }
    
    return $value;
  }

  /**
   * Check if a module is compatible with Drupal 11.
   *
   * @param string $name
   *   The module name.
   *
   * @return bool
   *   TRUE if the module is compatible, FALSE otherwise.
   */
  protected function isModuleCompatible($name) {
    // For now, just return FALSE to indicate all modules need fixing
    return FALSE;
  }

  /**
   * Check if a module has a known fix available.
   *
   * @param string $name
   *   The module name.
   *
   * @return bool
   *   TRUE if a fix is available, FALSE otherwise.
   */
  protected function hasKnownFix($name) {
    // For now, return TRUE for testing
    return TRUE;
  }

  /**
   * Returns the current rate limit status.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with rate limit information.
   */
  public function getRateLimitStatus() {
    return new JsonResponse($this->openai->getRateLimitStatus());
  }

}
