<?php

namespace Drupal\ai_upgrade_assistant\Service;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\State\StateInterface;

/**
 * Service for managing AI model configurations and fine-tuning.
 */
class AiModelManager {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The OpenAI service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\OpenAIService
   */
  protected $openai;

  /**
   * The community learning service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\CommunityLearningService
   */
  protected $communityLearning;

  /**
   * Constructs a new AiModelManager.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\ai_upgrade_assistant\Service\OpenAIService $openai
   *   The OpenAI service.
   * @param \Drupal\ai_upgrade_assistant\Service\CommunityLearningService $community_learning
   *   The community learning service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    CacheBackendInterface $cache,
    StateInterface $state,
    LoggerChannelFactoryInterface $logger_factory,
    OpenAIService $openai,
    CommunityLearningService $community_learning
  ) {
    $this->configFactory = $config_factory;
    $this->cache = $cache;
    $this->state = $state;
    $this->loggerFactory = $logger_factory;
    $this->openai = $openai;
    $this->communityLearning = $community_learning;
  }

  /**
   * Gets the optimal model configuration for a specific task.
   *
   * @param string $task_type
   *   The type of task (e.g., 'code_analysis', 'patch_generation').
   * @param array $context
   *   Additional context for the task.
   *
   * @return array
   *   Model configuration parameters.
   */
  public function getModelConfig($task_type, array $context = []) {
    $cid = 'ai_upgrade_assistant:model_config:' . $task_type;
    
    if ($cache = $this->cache->get($cid)) {
      return $cache->data;
    }

    $config = $this->configFactory->get('ai_upgrade_assistant.settings');
    $base_config = $config->get('ai_model_config.' . $task_type) ?: [];
    
    // Enhance with learning-based optimizations
    $optimized_config = $this->optimizeModelConfig($base_config, $task_type, $context);
    
    $this->cache->set($cid, $optimized_config, time() + 3600);
    return $optimized_config;
  }

  /**
   * Optimizes model configuration based on historical performance.
   *
   * @param array $base_config
   *   Base configuration to optimize.
   * @param string $task_type
   *   The type of task.
   * @param array $context
   *   Additional context.
   *
   * @return array
   *   Optimized configuration.
   */
  protected function optimizeModelConfig(array $base_config, $task_type, array $context) {
    $stats = $this->state->get('ai_upgrade_assistant.model_stats.' . $task_type, []);
    
    // Apply performance-based adjustments
    if (!empty($stats['success_patterns'])) {
      $base_config['temperature'] = $this->calculateOptimalTemperature($stats);
      $base_config['max_tokens'] = $this->calculateOptimalTokens($stats);
    }

    // Incorporate community patterns
    $patterns = $this->communityLearning->findSimilarPatterns($context);
    if (!empty($patterns)) {
      $base_config['examples'] = array_slice(
        array_column($patterns, 'pattern'),
        0,
        3
      );
    }

    return $base_config;
  }

  /**
   * Calculates optimal temperature based on success patterns.
   *
   * @param array $stats
   *   Model statistics.
   *
   * @return float
   *   Optimal temperature value.
   */
  protected function calculateOptimalTemperature(array $stats) {
    if (empty($stats['success_patterns'])) {
      return 0.7; // Default
    }

    // Calculate weighted average of successful temperatures
    $total_weight = 0;
    $weighted_sum = 0;

    foreach ($stats['success_patterns'] as $pattern) {
      $weight = $pattern['success_count'];
      $weighted_sum += $pattern['temperature'] * $weight;
      $total_weight += $weight;
    }

    return $total_weight > 0 ? $weighted_sum / $total_weight : 0.7;
  }

  /**
   * Calculates optimal token count based on success patterns.
   *
   * @param array $stats
   *   Model statistics.
   *
   * @return int
   *   Optimal token count.
   */
  protected function calculateOptimalTokens(array $stats) {
    if (empty($stats['success_patterns'])) {
      return 2048; // Default
    }

    // Find the most successful token count
    $token_counts = [];
    foreach ($stats['success_patterns'] as $pattern) {
      $tokens = $pattern['max_tokens'];
      if (!isset($token_counts[$tokens])) {
        $token_counts[$tokens] = 0;
      }
      $token_counts[$tokens] += $pattern['success_count'];
    }

    return array_search(max($token_counts), $token_counts) ?: 2048;
  }

  /**
   * Records the success or failure of a model configuration.
   *
   * @param string $task_type
   *   The type of task.
   * @param array $config
   *   The model configuration used.
   * @param bool $success
   *   Whether the task was successful.
   * @param array $metrics
   *   Performance metrics for the task.
   */
  public function recordModelPerformance($task_type, array $config, $success, array $metrics = []) {
    $stats = $this->state->get('ai_upgrade_assistant.model_stats.' . $task_type, [
      'success_patterns' => [],
      'failure_patterns' => [],
    ]);

    $pattern = $config + [
      'timestamp' => time(),
      'metrics' => $metrics,
    ];

    if ($success) {
      $stats['success_patterns'][] = $pattern;
    }
    else {
      $stats['failure_patterns'][] = $pattern;
    }

    // Keep only recent patterns
    $stats['success_patterns'] = array_slice($stats['success_patterns'], -100);
    $stats['failure_patterns'] = array_slice($stats['failure_patterns'], -100);

    $this->state->set('ai_upgrade_assistant.model_stats.' . $task_type, $stats);
    
    // Clear cached configurations
    $this->cache->delete('ai_upgrade_assistant:model_config:' . $task_type);
  }

}
