<?php

namespace Drupal\ai_upgrade_assistant\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\State\StateInterface;
use GuzzleHttp\ClientInterface;
use Drupal\ai_upgrade_assistant\Service\DataAnonymizer;

/**
 * Service for managing community learning and pattern sharing.
 */
class CommunityLearningService {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The data anonymizer service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\DataAnonymizer
   */
  protected $dataAnonymizer;

  /**
   * The Drupal.org API endpoint for patterns.
   *
   * @var string
   */
  protected const DRUPAL_PATTERNS_API = 'https://www.drupal.org/api-d7/ai_upgrade_patterns';

  /**
   * The GitLab project ID for pattern storage.
   *
   * @var string
   */
  protected const GITLAB_PROJECT_ID = 'project/ai_upgrade_patterns';

  /**
   * Constructs a new CommunityLearningService.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\ai_upgrade_assistant\Service\DataAnonymizer $data_anonymizer
   *   The data anonymizer service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    Connection $database,
    ClientInterface $http_client,
    LoggerChannelFactoryInterface $logger_factory,
    StateInterface $state,
    DataAnonymizer $data_anonymizer
  ) {
    $this->configFactory = $config_factory;
    $this->database = $database;
    $this->httpClient = $http_client;
    $this->loggerFactory = $logger_factory->get('ai_upgrade_assistant');
    $this->state = $state;
    $this->dataAnonymizer = $data_anonymizer;
  }

  /**
   * Stores a new upgrade pattern locally.
   *
   * @param array $pattern
   *   The upgrade pattern to store.
   *
   * @return int|null
   *   The ID of the stored pattern, or NULL if storage failed.
   */
  public function storePattern(array $pattern): ?int {
    try {
      // Store the original pattern for local reference.
      $id = $this->database->insert('ai_upgrade_patterns')
        ->fields([
          'type' => $pattern['type'],
          'module_name' => $pattern['module_name'],
          'from_version' => $pattern['from_version'],
          'to_version' => $pattern['to_version'],
          'pattern_data' => serialize($pattern),
          'context_data' => serialize($pattern['context'] ?? []),
          'success' => $pattern['success'] ?? 0,
          'sync_status' => 'pending',
          'created' => time(),
          'changed' => time(),
        ])
        ->execute();

      $this->loggerFactory->notice('Stored new upgrade pattern for @module', [
        '@module' => $pattern['module_name'],
      ]);

      return $id;
    }
    catch (\Exception $e) {
      $this->loggerFactory->error('Failed to store upgrade pattern: @error', [
        '@error' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Shares patterns with the Drupal.org public dataset.
   *
   * @return bool
   *   TRUE if patterns were shared successfully, FALSE otherwise.
   */
  public function sharePatterns(): bool {
    try {
      $patterns = $this->getLocalPatterns();
      if (empty($patterns)) {
        return TRUE;
      }

      // Anonymize patterns before sharing
      $anonymizedPatterns = array_map(function ($pattern) {
        return $this->dataAnonymizer->anonymizePattern($pattern);
      }, $patterns);

      // Add metadata for transparency
      foreach ($anonymizedPatterns as &$pattern) {
        $pattern['metadata'] = [
          'timestamp' => time(),
          'drupal_version' => \Drupal::VERSION,
          'module_version' => $this->getModuleVersion(),
          'success_rate' => $pattern['success_rate'] ?? 0,
          'community_votes' => $pattern['votes'] ?? 0,
          'source' => 'ai_upgrade_assistant',
          'maintainer' => $this->getDrupalOrgUsername(),
        ];
      }

      // Create a merge request with the new patterns
      $response = $this->httpClient->request('POST', 'https://git.drupal.org/api/v4/projects/' . self::GITLAB_PROJECT_ID . '/merge_requests', [
        'headers' => [
          'PRIVATE-TOKEN' => $this->getDrupalToken(),
          'Content-Type' => 'application/json',
        ],
        'json' => [
          'source_branch' => 'pattern-update-' . time(),
          'target_branch' => 'main',
          'title' => 'Add new upgrade patterns from community',
          'description' => $this->formatMergeRequestDescription($anonymizedPatterns),
          'patterns' => $anonymizedPatterns,
        ],
      ]);

      if ($response->getStatusCode() === 201) {
        $this->loggerFactory->info('Successfully created merge request with @count patterns', [
          '@count' => count($patterns),
        ]);
        return TRUE;
      }

      return FALSE;
    }
    catch (\Exception $e) {
      $this->loggerFactory->error('Failed to share patterns: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Gets the public dataset URL on Drupal.org.
   *
   * @return string
   *   The URL to the public dataset.
   */
  public function getPublicDatasetUrl(): string {
    return 'https://www.drupal.org/project/ai_upgrade_patterns';
  }

  /**
   * Gets community statistics for the patterns.
   *
   * @return array
   *   Array of statistics including total patterns, success rates, etc.
   */
  public function getDatasetStats(): array {
    try {
      $response = $this->httpClient->request('GET', self::DRUPAL_PATTERNS_API . '/stats');
      return json_decode($response->getBody(), TRUE);
    }
    catch (\Exception $e) {
      $this->loggerFactory->error('Failed to get dataset stats: @message', [
        '@message' => $e->getMessage(),
      ]);
      return [];
    }
  }

  /**
   * Formats a merge request description.
   *
   * @param array $patterns
   *   The patterns to include.
   *
   * @return string
   *   Formatted description.
   */
  protected function formatMergeRequestDescription(array $patterns): string {
    $description = "New upgrade patterns from the community:\n\n";
    foreach ($patterns as $pattern) {
      $description .= sprintf(
        "- Pattern for %s\n  - Success rate: %s%%\n  - Votes: %s\n  - Source: %s\n\n",
        $pattern['target'] ?? 'Unknown',
        $pattern['metadata']['success_rate'] ?? 0,
        $pattern['metadata']['community_votes'] ?? 0,
        $pattern['metadata']['source'] ?? 'Unknown'
      );
    }
    return $description;
  }

  /**
   * Gets local patterns.
   *
   * @return array
   *   Array of patterns.
   */
  private function getLocalPatterns(): array {
    $patterns = $this->database->select('ai_upgrade_patterns', 'p')
      ->fields('p')
      ->condition('sync_status', 'pending')
      ->execute()
      ->fetchAll();

    return array_map(function ($pattern) {
      return unserialize($pattern->pattern_data);
    }, $patterns);
  }

  /**
   * Gets the Drupal.org token.
   *
   * @return string
   *   The Drupal.org token.
   */
  private function getDrupalToken(): string {
    return $this->configFactory->get('ai_upgrade_assistant.settings')->get('drupal_token');
  }

  /**
   * Gets the Drupal.org username.
   *
   * @return string
   *   The Drupal.org username.
   */
  private function getDrupalOrgUsername(): string {
    return $this->configFactory->get('ai_upgrade_assistant.settings')->get('drupal_username');
  }

  /**
   * Gets the module version.
   *
   * @return string
   *   The module version.
   */
  private function getModuleVersion(): string {
    return \Drupal::VERSION;
  }
}
