<?php

namespace Drupal\ai_upgrade_assistant\Service\NodeVisitor;

use PhpParser\Node;
use PhpParser\NodeVisitorAbstract;

/**
 * Node visitor that detects class usage and inheritance.
 */
class ClassUsageVisitor extends NodeVisitorAbstract {

  /**
   * List of findings.
   *
   * @var array
   */
  protected $findings = [];

  /**
   * List of deprecated classes and their replacements.
   *
   * @var array
   */
  protected $deprecatedClasses = [
    'Drupal\Core\Entity\EntityNG' => [
      'replacement' => 'Drupal\Core\Entity\ContentEntityBase',
      'version' => '8.0.0',
    ],
    'Drupal\Core\Entity\EntityInterface' => [
      'replacement' => 'Drupal\Core\Entity\EntityInterface',
      'version' => '8.0.0',
    ],
    'Drupal\field\Plugin\Type\Widget\WidgetBase' => [
      'replacement' => 'Drupal\Core\Field\WidgetBase',
      'version' => '8.0.0',
    ],
  ];

  /**
   * {@inheritdoc}
   */
  public function enterNode(Node $node) {
    if ($node instanceof Node\Stmt\Class_) {
      // Check class inheritance
      if ($node->extends) {
        $extends = $node->extends->toString();
        $finding = [
          'type' => 'class_extends',
          'class' => $node->name->toString(),
          'extends' => $extends,
          'line' => $node->getLine(),
          'file' => $node->getAttribute('file'),
        ];

        if (isset($this->deprecatedClasses[$extends])) {
          $finding['deprecated'] = true;
          $finding['replacement'] = $this->deprecatedClasses[$extends]['replacement'];
          $finding['version'] = $this->deprecatedClasses[$extends]['version'];
        }

        $this->findings[] = $finding;
      }

      // Check interface implementations
      foreach ($node->implements as $interface) {
        $interfaceName = $interface->toString();
        $finding = [
          'type' => 'implements_interface',
          'class' => $node->name->toString(),
          'interface' => $interfaceName,
          'line' => $node->getLine(),
          'file' => $node->getAttribute('file'),
        ];

        if (isset($this->deprecatedClasses[$interfaceName])) {
          $finding['deprecated'] = true;
          $finding['replacement'] = $this->deprecatedClasses[$interfaceName]['replacement'];
          $finding['version'] = $this->deprecatedClasses[$interfaceName]['version'];
        }

        $this->findings[] = $finding;
      }
    }
    elseif ($node instanceof Node\Expr\New_) {
      // Check class instantiation
      if ($node->class instanceof Node\Name) {
        $className = $node->class->toString();
        $finding = [
          'type' => 'class_instantiation',
          'class' => $className,
          'line' => $node->getLine(),
          'file' => $node->getAttribute('file'),
        ];

        if (isset($this->deprecatedClasses[$className])) {
          $finding['deprecated'] = true;
          $finding['replacement'] = $this->deprecatedClasses[$className]['replacement'];
          $finding['version'] = $this->deprecatedClasses[$className]['version'];
        }

        $this->findings[] = $finding;
      }
    }
  }

  /**
   * Gets the findings.
   *
   * @return array
   *   Array of findings.
   */
  public function getFindings() {
    return $this->findings;
  }

  /**
   * Resets the findings.
   */
  public function reset() {
    $this->findings = [];
  }

}
