<?php

namespace Drupal\ai_upgrade_assistant\Service\NodeVisitor;

use PhpParser\Node;
use PhpParser\NodeVisitorAbstract;

/**
 * Node visitor that detects deprecated function calls.
 */
class DeprecatedFunctionVisitor extends NodeVisitorAbstract {

  /**
   * List of findings.
   *
   * @var array
   */
  protected $findings = [];

  /**
   * List of deprecated functions and their replacements.
   *
   * @var array
   */
  protected $deprecatedFunctions = [
    'drupal_set_message' => [
      'replacement' => '\Drupal::messenger()->addMessage()',
      'version' => '8.5.0',
    ],
    'format_date' => [
      'replacement' => '\Drupal::service("date.formatter")->format()',
      'version' => '8.0.0',
    ],
    'node_load' => [
      'replacement' => '\Drupal::entityTypeManager()->getStorage("node")->load()',
      'version' => '8.0.0',
    ],
    'user_load' => [
      'replacement' => '\Drupal::entityTypeManager()->getStorage("user")->load()',
      'version' => '8.0.0',
    ],
    'file_load' => [
      'replacement' => '\Drupal::entityTypeManager()->getStorage("file")->load()',
      'version' => '8.0.0',
    ],
    'entity_load' => [
      'replacement' => '\Drupal::entityTypeManager()->getStorage($entity_type)->load()',
      'version' => '8.0.0',
    ],
    'db_query' => [
      'replacement' => '\Drupal::database()->query()',
      'version' => '8.0.0',
    ],
    'db_select' => [
      'replacement' => '\Drupal::database()->select()',
      'version' => '8.0.0',
    ],
  ];

  /**
   * {@inheritdoc}
   */
  public function leaveNode(Node $node) {
    if ($node instanceof Node\Expr\FuncCall && $node->name instanceof Node\Name) {
      $name = $node->name->toString();
      if (isset($this->deprecatedFunctions[$name])) {
        $this->findings[] = [
          'type' => 'deprecated_function',
          'name' => $name,
          'line' => $node->getLine(),
          'replacement' => $this->deprecatedFunctions[$name]['replacement'],
          'version' => $this->deprecatedFunctions[$name]['version'],
          'file' => $node->getAttribute('file'),
        ];
      }
    }
    return null;
  }

  /**
   * Gets the findings.
   *
   * @return array
   *   Array of findings.
   */
  public function getFindings() {
    return $this->findings;
  }

  /**
   * Resets the findings.
   */
  public function reset() {
    $this->findings = [];
  }

}
