<?php

namespace Drupal\ai_upgrade_assistant\Service\NodeVisitor;

use PhpParser\Node;
use PhpParser\NodeVisitorAbstract;

/**
 * Node visitor that detects hook implementations.
 */
class HookVisitor extends NodeVisitorAbstract {

  /**
   * List of findings.
   *
   * @var array
   */
  protected $findings = [];

  /**
   * The module name.
   *
   * @var string
   */
  protected $moduleName;

  /**
   * List of deprecated hooks and their replacements.
   *
   * @var array
   */
  protected $deprecatedHooks = [
    'hook_entity_load' => [
      'replacement' => 'hook_ENTITY_TYPE_load',
      'version' => '8.7.0',
    ],
    'hook_entity_presave' => [
      'replacement' => 'hook_ENTITY_TYPE_presave',
      'version' => '8.7.0',
    ],
    'hook_entity_insert' => [
      'replacement' => 'hook_ENTITY_TYPE_insert',
      'version' => '8.7.0',
    ],
    'hook_entity_update' => [
      'replacement' => 'hook_ENTITY_TYPE_update',
      'version' => '8.7.0',
    ],
    'hook_entity_delete' => [
      'replacement' => 'hook_ENTITY_TYPE_delete',
      'version' => '8.7.0',
    ],
  ];

  /**
   * Constructs a new HookVisitor.
   *
   * @param string $module_name
   *   The module name.
   */
  public function __construct($module_name) {
    $this->moduleName = $module_name;
  }

  /**
   * {@inheritdoc}
   */
  public function enterNode(Node $node) {
    if ($node instanceof Node\Stmt\Function_) {
      $name = $node->name->toString();
      
      // Check if this is a hook implementation
      if (strpos($name, $this->moduleName . '_') === 0) {
        $hook_name = str_replace($this->moduleName . '_', 'hook_', $name);
        
        $finding = [
          'type' => 'hook',
          'name' => $name,
          'hook' => $hook_name,
          'line' => $node->getLine(),
          'file' => $node->getAttribute('file'),
          'params' => array_map(function($param) {
            return $param->var->name;
          }, $node->params),
        ];

        // Check if this is a deprecated hook
        if (isset($this->deprecatedHooks[$hook_name])) {
          $finding['deprecated'] = true;
          $finding['replacement'] = $this->deprecatedHooks[$hook_name]['replacement'];
          $finding['version'] = $this->deprecatedHooks[$hook_name]['version'];
        }

        $this->findings[] = $finding;
      }
    }
  }

  /**
   * Gets the findings.
   *
   * @return array
   *   Array of findings.
   */
  public function getFindings() {
    return $this->findings;
  }

  /**
   * Resets the findings.
   */
  public function reset() {
    $this->findings = [];
  }

}
