<?php

namespace Drupal\ai_upgrade_assistant\Service;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;

/**
 * Service for analyzing Drupal projects for upgrades using AI.
 */
class ProjectAnalyzer {
  use DependencySerializationTrait;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The OpenAI service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\OpenAIService
   */
  protected $openai;

  /**
   * The PHP parser service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\PhpParserService
   */
  protected $phpParser;

  /**
   * The AI model manager.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\AiModelManager
   */
  protected $aiModelManager;

  /**
   * The community learning service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\CommunityLearningService
   */
  protected $communityLearning;

  /**
   * Constructs a ProjectAnalyzer object.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\ai_upgrade_assistant\Service\OpenAIService $openai
   *   The OpenAI service.
   * @param \Drupal\ai_upgrade_assistant\Service\PhpParserService $php_parser
   *   The PHP parser service.
   * @param \Drupal\ai_upgrade_assistant\Service\AiModelManager $ai_model_manager
   *   The AI model manager service.
   * @param \Drupal\ai_upgrade_assistant\Service\CommunityLearningService $community_learning
   *   The community learning service.
   */
  public function __construct(
    ModuleHandlerInterface $module_handler,
    ConfigFactoryInterface $config_factory,
    FileSystemInterface $file_system,
    OpenAIService $openai,
    PhpParserService $php_parser,
    AiModelManager $ai_model_manager,
    CommunityLearningService $community_learning
  ) {
    $this->moduleHandler = $module_handler;
    $this->configFactory = $config_factory;
    $this->fileSystem = $file_system;
    $this->openai = $openai;
    $this->phpParser = $php_parser;
    $this->aiModelManager = $ai_model_manager;
    $this->communityLearning = $community_learning;
  }

  /**
   * Analyzes a Drupal project for upgrade requirements.
   *
   * @param string $project_path
   *   Path to the project to analyze.
   *
   * @return array
   *   Analysis results.
   */
  public function analyzeProject($project_path) {
    // Get optimal model configuration for code analysis
    $model_config = $this->aiModelManager->getModelConfig('code_analysis', [
      'project_path' => $project_path,
    ]);

    // Find similar patterns from community
    $patterns = $this->communityLearning->findSimilarPatterns([
      'project_path' => $project_path,
    ]);

    // Parse code and build AST
    $files = $this->findPhpFiles($project_path);
    $analysis_results = [];

    foreach ($files as $file) {
      $ast = $this->phpParser->parseFile($file);
      if (!$ast) {
        continue;
      }

      // Analyze code with AI
      $analysis = $this->analyzeCode($ast, $file, $model_config, $patterns);
      if ($analysis) {
        $analysis_results[$file] = $analysis;
      }
    }

    // Record successful patterns
    foreach ($analysis_results as $file => $result) {
      if (!empty($result['patterns'])) {
        foreach ($result['patterns'] as $pattern) {
          $this->communityLearning->recordPattern('code_analysis', $pattern, [
            'file' => $file,
          ]);
        }
      }
    }

    return $analysis_results;
  }

  /**
   * Analyzes code using AI and pattern matching.
   *
   * @param array $ast
   *   The AST of the code.
   * @param string $file
   *   The file being analyzed.
   * @param array $model_config
   *   AI model configuration.
   * @param array $patterns
   *   Similar patterns from community.
   *
   * @return array|null
   *   Analysis results or null if no issues found.
   */
  protected function analyzeCode(array $ast, $file, array $model_config, array $patterns) {
    $code_context = $this->buildCodeContext($ast, $file);
    
    // Enhance context with community patterns
    if (!empty($patterns)) {
      $code_context['similar_patterns'] = array_column($patterns, 'pattern');
    }

    // Get AI analysis
    $analysis = $this->openai->analyzeCode($code_context, $model_config);
    
    // Record model performance
    $this->aiModelManager->recordModelPerformance(
      'code_analysis',
      $model_config,
      !empty($analysis['suggestions']),
      [
        'response_time' => $analysis['metrics']['duration'] ?? 0,
        'token_count' => $analysis['metrics']['tokens'] ?? 0,
      ]
    );

    return $analysis;
  }

  /**
   * Builds context for code analysis.
   *
   * @param array $ast
   *   The AST of the code.
   * @param string $file
   *   The file being analyzed.
   *
   * @return array
   *   Code context for analysis.
   */
  protected function buildCodeContext(array $ast, $file) {
    return [
      'file_path' => $file,
      'ast' => $ast,
      'drupal_version' => \Drupal::VERSION,
      'module_info' => $this->getModuleInfo($file),
      'dependencies' => $this->getDependencies($file),
    ];
  }

  /**
   * Gets module info for a file.
   *
   * @param string $file
   *   File path.
   *
   * @return array
   *   Module information.
   */
  protected function getModuleInfo($file) {
    $module_name = basename(dirname($file));
    return $this->moduleHandler->getModule($module_name)->info ?? [];
  }

  /**
   * Gets dependencies for a file.
   *
   * @param string $file
   *   File path.
   *
   * @return array
   *   Dependencies information.
   */
  protected function getDependencies($file) {
    $module_name = basename(dirname($file));
    $module = $this->moduleHandler->getModule($module_name);
    
    return [
      'dependencies' => $module->info['dependencies'] ?? [],
      'required_by' => $this->moduleHandler->getRequiredBy($module_name),
    ];
  }

  /**
   * Finds all PHP files in a project.
   *
   * @param string $project_path
   *   Path to the project.
   *
   * @return array
   *   List of PHP files.
   */
  protected function findPhpFiles($project_path) {
    $files = [];
    $iterator = new \RecursiveIteratorIterator(
      new \RecursiveDirectoryIterator($project_path)
    );

    foreach ($iterator as $file) {
      if ($file->isFile() && $file->getExtension() === 'php') {
        $files[] = $file->getPathname();
      }
    }

    return $files;
  }

}
