<?php

namespace Drupal\Tests\ai_upgrade_assistant\Unit\Service;

use Drupal\ai_upgrade_assistant\Service\BatchAnalyzer;
use Drupal\ai_upgrade_assistant\Service\ProjectAnalyzer;
use Drupal\ai_upgrade_assistant\Service\AnalysisTracker;
use Drupal\ai_upgrade_assistant\Service\PatchSearcher;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Tests\UnitTestCase;

/**
 * @coversDefaultClass \Drupal\ai_upgrade_assistant\Service\BatchAnalyzer
 * @group ai_upgrade_assistant
 */
class BatchAnalyzerTest extends UnitTestCase {

  /**
   * The batch analyzer service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\BatchAnalyzer
   */
  protected $batchAnalyzer;

  /**
   * The mocked project analyzer.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\ProjectAnalyzer|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $projectAnalyzer;

  /**
   * The mocked state service.
   *
   * @var \Drupal\Core\State\StateInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $state;

  /**
   * The mocked module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $moduleHandler;

  /**
   * The mocked analysis tracker.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\AnalysisTracker|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $analysisTracker;

  /**
   * The mocked patch searcher.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\PatchSearcher|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $patchSearcher;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->projectAnalyzer = $this->createMock(ProjectAnalyzer::class);
    $this->state = $this->createMock(StateInterface::class);
    $this->moduleHandler = $this->createMock(ModuleHandlerInterface::class);
    $this->analysisTracker = $this->createMock(AnalysisTracker::class);
    $this->patchSearcher = $this->createMock(PatchSearcher::class);

    $this->batchAnalyzer = new BatchAnalyzer(
      $this->projectAnalyzer,
      $this->state,
      $this->moduleHandler,
      $this->analysisTracker,
      $this->patchSearcher
    );
  }

  /**
   * @covers ::createBatch
   */
  public function testCreateBatch() {
    $modules = ['test_module1', 'test_module2'];
    
    $batch = $this->batchAnalyzer->createBatch($modules);

    $this->assertIsArray($batch);
    $this->assertEquals('Analyzing modules', $batch['title']);
    $this->assertNotEmpty($batch['operations']);
    $this->assertTrue($batch['progressive']);
  }

  /**
   * @covers ::analyzeBatchItem
   */
  public function testAnalyzeBatchItem() {
    $module = 'test_module';
    $context = ['results' => [], 'message' => ''];

    $this->projectAnalyzer->expects($this->once())
      ->method('analyzeModule')
      ->with($module)
      ->willReturn(['status' => 'success']);

    $this->analysisTracker->expects($this->once())
      ->method('trackModuleAnalysis')
      ->with($module, ['status' => 'success']);

    $this->batchAnalyzer->analyzeBatchItem($module, &$context);

    $this->assertArrayHasKey($module, $context['results']);
    $this->assertEquals(['status' => 'success'], $context['results'][$module]);
  }

  /**
   * @covers ::analyzeBatchItem
   */
  public function testAnalyzeBatchItemWithError() {
    $module = 'test_module';
    $context = ['results' => [], 'message' => ''];

    $this->projectAnalyzer->expects($this->once())
      ->method('analyzeModule')
      ->with($module)
      ->willThrowException(new \Exception('Test error'));

    $this->analysisTracker->expects($this->once())
      ->method('trackModuleAnalysis')
      ->with($module, ['status' => 'error', 'message' => 'Test error']);

    $this->batchAnalyzer->analyzeBatchItem($module, &$context);

    $this->assertArrayHasKey($module, $context['results']);
    $this->assertEquals(['status' => 'error', 'message' => 'Test error'], $context['results'][$module]);
  }

  /**
   * @covers ::batchFinished
   */
  public function testBatchFinished() {
    $success = TRUE;
    $results = [
      'test_module1' => ['status' => 'success'],
      'test_module2' => ['status' => 'error', 'message' => 'Test error'],
    ];
    $operations = [];

    $this->state->expects($this->once())
      ->method('set')
      ->with('ai_upgrade_assistant.last_analysis_results', $results);

    $this->batchAnalyzer->batchFinished($success, $results, $operations);
  }

}
