<?php

namespace Drupal\Tests\ai_upgrade_assistant\Unit\Service;

use Drupal\ai_upgrade_assistant\Service\PatchSearcher;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\update\UpdateManagerInterface;
use GuzzleHttp\ClientInterface;
use Drupal\Tests\UnitTestCase;

/**
 * @coversDefaultClass \Drupal\ai_upgrade_assistant\Service\PatchSearcher
 * @group ai_upgrade_assistant
 */
class PatchSearcherTest extends UnitTestCase {

  /**
   * The patch searcher service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\PatchSearcher
   */
  protected $patchSearcher;

  /**
   * The mocked update manager.
   *
   * @var \Drupal\update\UpdateManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $updateManager;

  /**
   * The mocked module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $moduleHandler;

  /**
   * The mocked HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $httpClient;

  /**
   * The mocked file system.
   *
   * @var \Drupal\Core\File\FileSystemInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $fileSystem;

  /**
   * The mocked logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $loggerFactory;

  /**
   * The mocked state service.
   *
   * @var \Drupal\Core\State\StateInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $state;

  /**
   * The mocked cache service.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $cache;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->updateManager = $this->createMock(UpdateManagerInterface::class);
    $this->moduleHandler = $this->createMock(ModuleHandlerInterface::class);
    $this->httpClient = $this->createMock(ClientInterface::class);
    $this->fileSystem = $this->createMock(FileSystemInterface::class);
    $this->loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);
    $this->state = $this->createMock(StateInterface::class);
    $this->cache = $this->createMock(CacheBackendInterface::class);

    $this->patchSearcher = new PatchSearcher(
      $this->updateManager,
      $this->moduleHandler,
      $this->httpClient,
      $this->fileSystem,
      $this->loggerFactory,
      $this->state,
      $this->cache
    );
  }

  /**
   * @covers ::searchPatches
   */
  public function testSearchPatches() {
    $module = 'test_module';
    $version = '8.x-1.0';
    $issue_data = [
      'field_issue_status' => 8,
      'field_issue_version' => $version,
      'field_issue_component' => 'test_module',
      'field_project' => ['uri' => 'test_module'],
      'field_issue_files' => [
        ['uri' => 'https://drupal.org/files/test.patch'],
      ],
    ];

    $this->moduleHandler->expects($this->once())
      ->method('getModule')
      ->with($module)
      ->willReturn((object) ['info' => ['version' => $version]]);

    $this->cache->expects($this->once())
      ->method('get')
      ->with('ai_upgrade_assistant_patches_' . $module)
      ->willReturn(FALSE);

    // Mock the API response
    $response = $this->createMock(\Psr\Http\Message\ResponseInterface::class);
    $response->expects($this->once())
      ->method('getBody')
      ->willReturn(json_encode(['list' => [$issue_data]]));

    $this->httpClient->expects($this->once())
      ->method('request')
      ->willReturn($response);

    $patches = $this->patchSearcher->searchPatches($module);

    $this->assertIsArray($patches);
    $this->assertNotEmpty($patches);
    $this->assertEquals('https://drupal.org/files/test.patch', $patches[0]['patch_url']);
  }

  /**
   * @covers ::downloadPatch
   */
  public function testDownloadPatch() {
    $patch_url = 'https://drupal.org/files/test.patch';
    $patch_content = 'Test patch content';
    $temp_path = '/tmp/test.patch';

    // Mock the API response
    $response = $this->createMock(\Psr\Http\Message\ResponseInterface::class);
    $response->expects($this->once())
      ->method('getBody')
      ->willReturn($patch_content);

    $this->httpClient->expects($this->once())
      ->method('request')
      ->with('GET', $patch_url)
      ->willReturn($response);

    $this->fileSystem->expects($this->once())
      ->method('tempnam')
      ->willReturn($temp_path);

    $this->fileSystem->expects($this->once())
      ->method('saveData')
      ->with($patch_content, $temp_path)
      ->willReturn(strlen($patch_content));

    $result = $this->patchSearcher->downloadPatch($patch_url);

    $this->assertEquals($temp_path, $result);
  }

  /**
   * @covers ::downloadPatch
   */
  public function testDownloadPatchFailure() {
    $patch_url = 'https://drupal.org/files/test.patch';

    $this->httpClient->expects($this->once())
      ->method('request')
      ->willThrowException(new \Exception('Download failed'));

    $this->expectException(\Exception::class);
    $this->expectExceptionMessage('Failed to download patch: Download failed');

    $this->patchSearcher->downloadPatch($patch_url);
  }

}
