<?php

namespace Drupal\Tests\ai_upgrade_assistant\Unit\Service\ProjectAnalyzer;

use Drupal\ai_upgrade_assistant\Service\ProjectAnalyzer;
use Drupal\ai_upgrade_assistant\Service\OpenAIService;
use Drupal\ai_upgrade_assistant\Service\PhpParserService;
use Drupal\ai_upgrade_assistant\Service\AiPromptGenerator;
use Drupal\Tests\UnitTestCase;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\upgrade_status\DeprecationAnalyzer;

/**
 * Tests the project analyzer service.
 *
 * @group ai_upgrade_assistant
 * @coversDefaultClass \Drupal\ai_upgrade_assistant\Service\ProjectAnalyzer
 */
class ProjectAnalyzerTest extends UnitTestCase {

  /**
   * The project analyzer service under test.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\ProjectAnalyzer
   */
  protected $projectAnalyzer;

  /**
   * The mock module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The mock config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The mock file system.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The mock OpenAI service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\OpenAIService
   */
  protected $openAIService;

  /**
   * The mock deprecation analyzer.
   *
   * @var \Drupal\upgrade_status\DeprecationAnalyzer
   */
  protected $deprecationAnalyzer;

  /**
   * The mock PHP parser service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\PhpParserService
   */
  protected $phpParser;

  /**
   * The mock AI prompt generator.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\AiPromptGenerator
   */
  protected $promptGenerator;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->moduleHandler = $this->createMock(ModuleHandlerInterface::class);
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->fileSystem = $this->createMock(FileSystemInterface::class);
    $this->openAIService = $this->createMock(OpenAIService::class);
    $this->deprecationAnalyzer = $this->createMock(DeprecationAnalyzer::class);
    $this->phpParser = $this->createMock(PhpParserService::class);
    $this->promptGenerator = $this->createMock(AiPromptGenerator::class);

    $this->projectAnalyzer = new ProjectAnalyzer(
      $this->moduleHandler,
      $this->configFactory,
      $this->fileSystem,
      $this->openAIService,
      $this->deprecationAnalyzer,
      $this->phpParser,
      $this->promptGenerator
    );
  }

  /**
   * Tests module detection.
   *
   * @covers ::analyzeModule
   */
  public function testModuleDetection() {
    $module_path = '/path/to/test_module';
    $info_file_content = <<<EOT
name: Test Module
type: module
description: 'A test module'
core_version_requirement: ^9 || ^10
EOT;

    $this->fileSystem->expects($this->once())
      ->method('scanDirectory')
      ->willReturn([
        'test_module.info.yml' => new \stdClass(),
        'test_module.module' => new \stdClass(),
        'src/TestClass.php' => new \stdClass(),
      ]);

    $this->fileSystem->expects($this->any())
      ->method('readFile')
      ->willReturn($info_file_content);

    $result = $this->projectAnalyzer->analyzeModule('test_module', $module_path);
    
    $this->assertArrayHasKey('name', $result);
    $this->assertArrayHasKey('files', $result);
    $this->assertArrayHasKey('compatibility', $result);
  }

  /**
   * Tests AST parsing integration.
   *
   * @covers ::analyzeCode
   */
  public function testAstParsing() {
    $test_code = '<?php function test_function() { return "test"; }';
    
    $this->phpParser->expects($this->once())
      ->method('parseCode')
      ->with($test_code)
      ->willReturn([
        'functions' => ['test_function'],
        'classes' => [],
        'deprecated' => [],
      ]);

    $result = $this->projectAnalyzer->analyzeCode($test_code);
    
    $this->assertArrayHasKey('functions', $result);
    $this->assertArrayHasKey('classes', $result);
    $this->assertArrayHasKey('deprecated', $result);
  }

  /**
   * Tests dependency analysis.
   *
   * @covers ::analyzeDependencies
   */
  public function testDependencyAnalysis() {
    $dependencies = [
      'core' => '^9 || ^10',
      'dependencies' => [
        'views',
        'node',
      ],
    ];

    $this->moduleHandler->expects($this->any())
      ->method('moduleExists')
      ->willReturn(true);

    $result = $this->projectAnalyzer->analyzeDependencies($dependencies);
    
    $this->assertArrayHasKey('core_compatible', $result);
    $this->assertArrayHasKey('missing_dependencies', $result);
    $this->assertEmpty($result['missing_dependencies']);
  }

  /**
   * Tests version compatibility checking.
   *
   * @covers ::checkVersionCompatibility
   */
  public function testVersionCompatibilityCheck() {
    $module_info = [
      'core_version_requirement' => '^9 || ^10',
      'php' => '8.1',
    ];

    $result = $this->projectAnalyzer->checkVersionCompatibility($module_info);
    
    $this->assertArrayHasKey('drupal_compatible', $result);
    $this->assertArrayHasKey('php_compatible', $result);
    $this->assertTrue($result['drupal_compatible']);
  }

  /**
   * Tests AI integration for code analysis.
   *
   * @covers ::getAiAnalysis
   */
  public function testAiCodeAnalysis() {
    $code_sample = '<?php function deprecated_function() { }';
    
    $this->promptGenerator->expects($this->once())
      ->method('generateCodeAnalysisPrompt')
      ->willReturn('Analyze this code...');

    $this->openAIService->expects($this->once())
      ->method('generateCompletion')
      ->willReturn('This function is deprecated...');

    $result = $this->projectAnalyzer->getAiAnalysis($code_sample);
    
    $this->assertNotEmpty($result);
    $this->assertIsString($result);
  }
}
