<?php

namespace Drupal\ai_upgrade_assistant\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Exception\ConnectException;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\ServerException;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\State\StateInterface;

/**
 * Service for interacting with OpenAI API.
 */
class OpenAIService {

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * Whether we're in test mode.
   *
   * @var bool
   */
  protected $testMode = FALSE;

  /**
   * Maximum retries for API calls.
   *
   * @var int
   */
  protected $maxRetries = 3;

  /**
   * The rate limit window in seconds.
   *
   * @var int
   */
  protected $rateLimitWindow;

  /**
   * Maximum requests per window.
   *
   * @var int
   */
  protected $maxRequestsPerWindow;

  /**
   * Whether rate limiting is enabled.
   *
   * @var bool
   */
  protected $rateLimitEnabled;

  /**
   * Base delay between retries in seconds.
   *
   * @var int
   */
  protected $baseDelay = 1;

  /**
   * Constructs a new OpenAIService.
   *
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   */
  public function __construct(
    ClientInterface $http_client,
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory,
    CacheBackendInterface $cache,
    StateInterface $state
  ) {
    $this->httpClient = $http_client;
    $this->configFactory = $config_factory;
    $this->loggerFactory = $logger_factory;
    $this->cache = $cache;
    $this->state = $state;
    
    // Load rate limit settings
    $config = $this->configFactory->get('ai_upgrade_assistant.settings');
    $this->rateLimitWindow = $config->get('rate_limit.window_size') ?? 60;
    $this->maxRequestsPerWindow = $config->get('rate_limit.max_requests') ?? 50;
    $this->rateLimitEnabled = $config->get('rate_limit.enabled') ?? true;
  }

  /**
   * Checks if we've hit the rate limit.
   *
   * @return bool
   *   TRUE if rate limited, FALSE otherwise.
   */
  protected function isRateLimited() {
    if (!$this->rateLimitEnabled) {
      return FALSE;
    }

    $window_key = 'openai_rate_limit_window';
    $count_key = 'openai_request_count';
    
    $window_start = $this->state->get($window_key);
    $current_time = time();

    // Start new window if needed
    if (!$window_start || $current_time - $window_start >= $this->rateLimitWindow) {
      $this->state->set($window_key, $current_time);
      $this->state->set($count_key, 0);
      return FALSE;
    }

    $request_count = $this->state->get($count_key, 0);
    
    if ($request_count >= $this->maxRequestsPerWindow) {
      $this->loggerFactory->get('ai_upgrade_assistant')
        ->warning('Rate limit reached: @count requests in @window seconds', [
          '@count' => $request_count,
          '@window' => $this->rateLimitWindow,
        ]);
      return TRUE;
    }

    return FALSE;
  }

  /**
   * Increments the request counter.
   */
  protected function incrementRequestCount() {
    $count_key = 'openai_request_count';
    $current_count = $this->state->get($count_key, 0);
    $this->state->set($count_key, $current_count + 1);
  }

  /**
   * Makes an API call with retry logic and rate limiting.
   *
   * @param string $endpoint
   *   The API endpoint.
   * @param array $data
   *   The request data.
   * @param string $method
   *   The HTTP method.
   *
   * @return array|null
   *   The API response or null on failure.
   *
   * @throws \Exception
   *   When the API call fails after all retries.
   */
  protected function makeApiCall($endpoint, array $data, $method = 'POST') {
    if ($this->isRateLimited()) {
      throw new \Exception('Rate limit exceeded. Please try again later.');
    }

    $config = $this->configFactory->get('ai_upgrade_assistant.settings');
    $api_key = $config->get('openai_api_key');

    if (empty($api_key)) {
      $this->loggerFactory->get('ai_upgrade_assistant')->error('OpenAI API key is not configured');
      throw new \Exception('OpenAI API key is not configured');
    }

    $retry_count = 0;
    $last_exception = NULL;

    while ($retry_count < $this->maxRetries) {
      try {
        if ($retry_count > 0) {
          // Exponential backoff
          $delay = $this->baseDelay * pow(2, $retry_count);
          sleep($delay);
        }

        $response = $this->httpClient->request($method, $endpoint, [
          'json' => $data,
          'headers' => [
            'Authorization' => 'Bearer ' . $api_key,
            'Content-Type' => 'application/json',
          ],
        ]);

        $this->incrementRequestCount();
        
        return json_decode((string) $response->getBody(), TRUE);

      }
      catch (ClientException $e) {
        // 4xx errors - client errors
        $this->loggerFactory->get('ai_upgrade_assistant')->error(
          'OpenAI API client error: @message', ['@message' => $e->getMessage()]
        );
        throw $e;
      }
      catch (ServerException $e) {
        // 5xx errors - server errors, retry
        $last_exception = $e;
        $this->loggerFactory->get('ai_upgrade_assistant')->warning(
          'OpenAI API server error (attempt @attempt): @message',
          ['@attempt' => $retry_count + 1, '@message' => $e->getMessage()]
        );
      }
      catch (ConnectException $e) {
        // Connection errors, retry
        $last_exception = $e;
        $this->loggerFactory->get('ai_upgrade_assistant')->warning(
          'OpenAI API connection error (attempt @attempt): @message',
          ['@attempt' => $retry_count + 1, '@message' => $e->getMessage()]
        );
      }
      catch (RequestException $e) {
        // Other request errors, retry
        $last_exception = $e;
        $this->loggerFactory->get('ai_upgrade_assistant')->warning(
          'OpenAI API request error (attempt @attempt): @message',
          ['@attempt' => $retry_count + 1, '@message' => $e->getMessage()]
        );
      }

      $retry_count++;
    }

    // If we get here, all retries failed
    $this->loggerFactory->get('ai_upgrade_assistant')->error(
      'OpenAI API call failed after @max_retries retries. Last error: @message',
      ['@max_retries' => $this->maxRetries, '@message' => $last_exception->getMessage()]
    );

    throw new \Exception(
      'Failed to communicate with OpenAI API after multiple attempts. Please try again later.',
      0,
      $last_exception
    );
  }

  /**
   * Analyzes code using OpenAI's API.
   *
   * @param string $code
   *   The code to analyze.
   * @param array $options
   *   Additional options for the analysis.
   *
   * @return array
   *   The analysis results.
   */
  public function analyzeCode($code, array $options = []) {
    try {
      $data = [
        'model' => 'gpt-4',
        'messages' => [
          [
            'role' => 'system',
            'content' => 'You are a Drupal expert analyzing code for compatibility issues.',
          ],
          [
            'role' => 'user',
            'content' => "Please analyze this code for Drupal compatibility issues:\n\n$code",
          ],
        ],
        'temperature' => 0.7,
        'max_tokens' => 2000,
      ];

      $response = $this->makeApiCall('https://api.openai.com/v1/chat/completions', $data);

      if (!empty($response['choices'][0]['message']['content'])) {
        return [
          'status' => 'success',
          'analysis' => $response['choices'][0]['message']['content'],
        ];
      }

      return [
        'status' => 'error',
        'message' => 'Invalid response format from OpenAI API',
      ];
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('ai_upgrade_assistant')->error(
        'Error analyzing code: @message',
        ['@message' => $e->getMessage()]
      );

      return [
        'status' => 'error',
        'message' => $e->getMessage(),
      ];
    }
  }

  /**
   * Generates a patch for the given code changes.
   *
   * @param string $original_code
   *   The original code.
   * @param array $issues
   *   The issues to fix.
   *
   * @return array
   *   The generated patch and metadata.
   */
  public function generatePatch($original_code, array $issues) {
    try {
      $prompt = $this->buildPatchPrompt($original_code, $issues);
      
      $data = [
        'model' => 'gpt-4',
        'messages' => [
          [
            'role' => 'system',
            'content' => 'You are a Drupal expert generating patches to fix compatibility issues.',
          ],
          [
            'role' => 'user',
            'content' => $prompt,
          ],
        ],
        'temperature' => 0.3,
        'max_tokens' => 2000,
      ];

      $response = $this->makeApiCall('https://api.openai.com/v1/chat/completions', $data);

      if (!empty($response['choices'][0]['message']['content'])) {
        return [
          'status' => 'success',
          'patch' => $response['choices'][0]['message']['content'],
        ];
      }

      return [
        'status' => 'error',
        'message' => 'Invalid response format from OpenAI API',
      ];
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('ai_upgrade_assistant')->error(
        'Error generating patch: @message',
        ['@message' => $e->getMessage()]
      );

      return [
        'status' => 'error',
        'message' => $e->getMessage(),
      ];
    }
  }

  /**
   * Builds a prompt for patch generation.
   *
   * @param string $original_code
   *   The original code.
   * @param array $issues
   *   The issues to fix.
   *
   * @return string
   *   The constructed prompt.
   */
  protected function buildPatchPrompt($original_code, array $issues) {
    $prompt = "Please generate a patch to fix the following Drupal compatibility issues:\n\n";
    $prompt .= "Original code:\n```php\n$original_code\n```\n\n";
    $prompt .= "Issues to fix:\n";
    
    foreach ($issues as $issue) {
      $prompt .= "- {$issue['description']}\n";
    }
    
    $prompt .= "\nPlease provide the patch in unified diff format.";
    
    return $prompt;
  }

  /**
   * Gets the current rate limit status.
   *
   * @return array
   *   An array containing rate limit information.
   */
  public function getRateLimitStatus() {
    $window_key = 'openai_rate_limit_window';
    $count_key = 'openai_request_count';
    
    $window_start = $this->state->get($window_key);
    $request_count = $this->state->get($count_key, 0);
    $current_time = time();

    return [
      'enabled' => $this->rateLimitEnabled,
      'window_size' => $this->rateLimitWindow,
      'max_requests' => $this->maxRequestsPerWindow,
      'current_requests' => $request_count,
      'window_remaining' => $window_start ? max(0, $this->rateLimitWindow - ($current_time - $window_start)) : $this->rateLimitWindow,
      'requests_remaining' => max(0, $this->maxRequestsPerWindow - $request_count),
    ];
  }

  /**
   * Sets test mode for the service.
   *
   * @param bool $mode
   *   The test mode value.
   */
  public function setTestMode($mode) {
    $this->testMode = (bool) $mode;
  }

}
