<?php

namespace Drupal\ai_upgrade_assistant\Service;

use PhpParser\Error;
use PhpParser\NodeTraverser;
use PhpParser\ParserFactory;
use PhpParser\Node;
use PhpParser\NodeVisitor\NameResolver;
use PhpParser\NodeFinder;
use PhpParser\PrettyPrinter;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\ai_upgrade_assistant\Service\NodeVisitor\DeprecatedFunctionVisitor;
use Drupal\ai_upgrade_assistant\Service\NodeVisitor\HookVisitor;
use Drupal\ai_upgrade_assistant\Service\NodeVisitor\ClassUsageVisitor;

/**
 * Service for parsing and analyzing PHP code using PHP Parser.
 */
class PhpParserService {

  /**
   * The PHP parser instance.
   *
   * @var \PhpParser\Parser
   */
  protected $parser;

  /**
   * The node traverser.
   *
   * @var \PhpParser\NodeTraverser
   */
  protected $traverser;

  /**
   * The node finder.
   *
   * @var \PhpParser\NodeFinder
   */
  protected $nodeFinder;

  /**
   * The pretty printer.
   *
   * @var \PhpParser\PrettyPrinter\Standard
   */
  protected $prettyPrinter;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The deprecated function visitor.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\NodeVisitor\DeprecatedFunctionVisitor
   */
  protected $deprecatedFunctionVisitor;

  /**
   * The hook visitor.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\NodeVisitor\HookVisitor
   */
  protected $hookVisitor;

  /**
   * The class usage visitor.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\NodeVisitor\ClassUsageVisitor
   */
  protected $classUsageVisitor;

  /**
   * Constructs a new PhpParserService.
   *
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory service.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   */
  public function __construct(
    LoggerChannelFactoryInterface $logger_factory,
    FileSystemInterface $file_system
  ) {
    $this->loggerFactory = $logger_factory;
    $this->fileSystem = $file_system;
    
    // Initialize parser with PHP 7 support
    $this->parser = (new ParserFactory)->create(ParserFactory::ONLY_PHP7);
    $this->traverser = new NodeTraverser();
    $this->traverser->addVisitor(new NameResolver());
    
    // Initialize visitors
    $this->deprecatedFunctionVisitor = new DeprecatedFunctionVisitor();
    $this->classUsageVisitor = new ClassUsageVisitor();
    
    $this->nodeFinder = new NodeFinder();
    $this->prettyPrinter = new PrettyPrinter\Standard();
  }

  /**
   * Parses a PHP file and returns its AST.
   *
   * @param string $file_path
   *   Path to the PHP file.
   *
   * @return \PhpParser\Node\Stmt[]|null
   *   Array of statements or null on error.
   */
  public function parseFile(string $file_path) {
    try {
      $code = file_get_contents($file_path);
      if ($code === FALSE) {
        $this->loggerFactory->get('ai_upgrade_assistant')->error(
          'Failed to read file: @file',
          ['@file' => $file_path]
        );
        return NULL;
      }
      return $this->parser->parse($code);
    }
    catch (Error $error) {
      $this->loggerFactory->get('ai_upgrade_assistant')->error(
        'Parse error in @file: @message',
        [
          '@file' => $file_path,
          '@message' => $error->getMessage(),
        ]
      );
      return NULL;
    }
  }

  /**
   * Analyzes a PHP file for deprecated code and compatibility issues.
   *
   * @param string $file_path
   *   Path to the PHP file.
   * @param string $module_name
   *   The module name.
   *
   * @return array
   *   Analysis results containing deprecated functions, hooks, and class usage.
   */
  public function analyzeFile(string $file_path, string $module_name) {
    $ast = $this->parseFile($file_path);
    if (!$ast) {
      return [
        'status' => 'error',
        'message' => 'Failed to parse file',
      ];
    }

    // Reset visitors
    $this->deprecatedFunctionVisitor->reset();
    $this->classUsageVisitor->reset();

    // Create a new hook visitor for this module
    $this->hookVisitor = new HookVisitor($module_name);

    // Add visitors to traverser
    $traverser = new NodeTraverser();
    $traverser->addVisitor(new NameResolver());
    $traverser->addVisitor($this->deprecatedFunctionVisitor);
    $traverser->addVisitor($this->hookVisitor);
    $traverser->addVisitor($this->classUsageVisitor);

    // Traverse the AST
    $traverser->traverse($ast);

    return [
      'status' => 'success',
      'file' => $file_path,
      'deprecated_functions' => $this->deprecatedFunctionVisitor->getFindings(),
      'hooks' => $this->hookVisitor->getFindings(),
      'class_usage' => $this->classUsageVisitor->getFindings(),
    ];
  }

  /**
   * Gets all deprecated items from a file.
   *
   * @param string $file_path
   *   Path to the PHP file.
   * @param string $module_name
   *   The module name.
   *
   * @return array
   *   Array of deprecated items with their replacements.
   */
  public function getDeprecatedItems(string $file_path, string $module_name) {
    $analysis = $this->analyzeFile($file_path, $module_name);
    if ($analysis['status'] !== 'success') {
      return [];
    }

    $deprecated_items = [];

    // Add deprecated functions
    foreach ($analysis['deprecated_functions'] as $function) {
      $deprecated_items[] = [
        'type' => 'function',
        'name' => $function['name'],
        'replacement' => $function['replacement'],
        'version' => $function['version'],
        'line' => $function['line'],
        'file' => $file_path,
      ];
    }

    // Add deprecated hooks
    foreach ($analysis['hooks'] as $hook) {
      if (!empty($hook['deprecated'])) {
        $deprecated_items[] = [
          'type' => 'hook',
          'name' => $hook['name'],
          'replacement' => $hook['replacement'],
          'version' => $hook['version'],
          'line' => $hook['line'],
          'file' => $file_path,
        ];
      }
    }

    // Add deprecated classes
    foreach ($analysis['class_usage'] as $usage) {
      if (!empty($usage['deprecated'])) {
        $deprecated_items[] = [
          'type' => $usage['type'],
          'name' => $usage['class'],
          'replacement' => $usage['replacement'],
          'version' => $usage['version'],
          'line' => $usage['line'],
          'file' => $file_path,
        ];
      }
    }

    return $deprecated_items;
  }

}
