<?php

namespace Drupal\Tests\ai_upgrade_assistant\Kernel\Service;

use Drupal\KernelTests\KernelTestBase;
use Drupal\ai_upgrade_assistant\Service\OpenAIService;

/**
 * Kernel tests for the OpenAI service.
 *
 * @group ai_upgrade_assistant
 * @coversDefaultClass \Drupal\ai_upgrade_assistant\Service\OpenAIService
 */
class OpenAIServiceTest extends KernelTestBase {

  /**
   * The modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'system',
    'ai_upgrade_assistant',
  ];

  /**
   * The OpenAI service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\OpenAIService
   */
  protected $openAiService;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installConfig(['ai_upgrade_assistant']);
    
    // Get the service from the container
    $this->openAiService = $this->container->get('ai_upgrade_assistant.openai');
    
    // Enable test mode to avoid real API calls
    $this->openAiService->setTestMode(TRUE);
  }

  /**
   * Tests the service can be instantiated from the container.
   */
  public function testServiceInstantiation() {
    $this->assertInstanceOf(OpenAIService::class, $this->openAiService);
  }

  /**
   * Tests the analysis workflow with configuration.
   */
  public function testAnalysisWithConfiguration() {
    // Set a test API key in configuration
    $this->config('ai_upgrade_assistant.settings')
      ->set('openai_api_key', 'sk-testkey123456789012345678901234567890')
      ->save();

    $code = '<?php function test() { return TRUE; }';
    $result = $this->openAiService->analyzeCode($code, ['module' => 'test']);

    $this->assertIsArray($result);
    $this->assertArrayHasKey('severity', $result);
    $this->assertArrayHasKey('explanation', $result);
    $this->assertArrayHasKey('suggestions', $result);
    $this->assertArrayHasKey('next_steps', $result);
  }

  /**
   * Tests analysis with various code samples.
   *
   * @dataProvider codeExamplesProvider
   */
  public function testAnalysisWithVariousCodeSamples($code, $context) {
    $result = $this->openAiService->analyzeCode($code, $context);
    
    $this->assertNotEmpty($result['severity']);
    $this->assertContains($result['severity'], ['critical', 'high', 'medium', 'low']);
    $this->assertNotEmpty($result['suggestions']);
  }

  /**
   * Data provider for testAnalysisWithVariousCodeSamples.
   */
  public function codeExamplesProvider() {
    return [
      'simple_function' => [
        '<?php function example() { return TRUE; }',
        ['module' => 'test', 'type' => 'function'],
      ],
      'deprecated_function' => [
        '<?php drupal_set_message("Test");',
        ['module' => 'test', 'type' => 'deprecated'],
      ],
      'class_definition' => [
        '<?php class TestController extends ControllerBase { }',
        ['module' => 'test', 'type' => 'class'],
      ],
    ];
  }

}
