<?php

namespace Drupal\Tests\ai_upgrade_assistant\Unit\Service;

use Drupal\ai_upgrade_assistant\Service\ProjectAnalyzer;
use Drupal\ai_upgrade_assistant\Service\OpenAIService;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\File\FileSystemInterface;
use Drupal\upgrade_status\DeprecationAnalyzer;
use Drupal\Tests\UnitTestCase;
use Prophecy\Argument;

/**
 * Unit tests for the ProjectAnalyzer service.
 *
 * @group ai_upgrade_assistant
 * @coversDefaultClass \Drupal\ai_upgrade_assistant\Service\ProjectAnalyzer
 */
class ProjectAnalyzerTest extends UnitTestCase {

  /**
   * The project analyzer service under test.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\ProjectAnalyzer
   */
  protected $projectAnalyzer;

  /**
   * The module handler prophecy.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $moduleHandler;

  /**
   * The config factory prophecy.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $configFactory;

  /**
   * The file system prophecy.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $fileSystem;

  /**
   * The OpenAI service prophecy.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $openai;

  /**
   * The deprecation analyzer prophecy.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $deprecationAnalyzer;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create prophecies for dependencies
    $this->moduleHandler = $this->prophesize(ModuleHandlerInterface::class);
    $this->configFactory = $this->prophesize(ConfigFactoryInterface::class);
    $this->fileSystem = $this->prophesize(FileSystemInterface::class);
    $this->openai = $this->prophesize(OpenAIService::class);
    $this->deprecationAnalyzer = $this->prophesize(DeprecationAnalyzer::class);

    // Setup config
    $config = $this->prophesize(ImmutableConfig::class);
    $this->configFactory->get('ai_upgrade_assistant.settings')
      ->willReturn($config->reveal());

    // Create service instance
    $this->projectAnalyzer = new ProjectAnalyzer(
      $this->moduleHandler->reveal(),
      $this->configFactory->reveal(),
      $this->fileSystem->reveal(),
      $this->openai->reveal(),
      $this->deprecationAnalyzer->reveal()
    );
  }

  /**
   * Tests getting project information.
   *
   * @covers ::getProjectInfo
   */
  public function testGetProjectInfo() {
    // Mock module list
    $modules = [
      'test_module' => $this->getMockExtension('test_module', [
        'package' => 'Custom',
        'name' => 'Test Module',
        'version' => '8.x-1.0',
      ]),
      'system' => $this->getMockExtension('system', [
        'package' => 'Core',
        'name' => 'System',
        'version' => '9.5.0',
      ]),
    ];

    $this->moduleHandler->getModuleList()->willReturn($modules);
    
    // Test getting project info
    $info = $this->projectAnalyzer->getProjectInfo();

    $this->assertArrayHasKey('drupal_version', $info);
    $this->assertArrayHasKey('installed_modules', $info);
    $this->assertArrayHasKey('custom_modules', $info);
    $this->assertArrayHasKey('test_module', $info['custom_modules']);
    $this->assertArrayHasKey('system', $info['installed_modules']);
  }

  /**
   * Tests analyzing a custom module.
   *
   * @covers ::analyzeCustomModule
   */
  public function testAnalyzeCustomModule() {
    $module_name = 'test_module';
    $module_path = '/path/to/test_module';
    $test_file = $module_path . '/test_module.module';

    // Mock file operations
    $this->fileSystem->scanDirectory(Argument::any(), Argument::any())
      ->willReturn([$test_file => (object) ['uri' => $test_file]]);

    // Mock OpenAI analysis
    $this->openai->analyzeCode(Argument::any(), Argument::any())
      ->willReturn([
        'severity' => 'medium',
        'explanation' => 'Test explanation',
        'suggestions' => ['Test suggestion'],
        'next_steps' => 'Test next steps',
      ]);

    // Test module analysis
    $results = $this->projectAnalyzer->analyzeCustomModule($module_name, $module_path);

    $this->assertArrayHasKey('files_analyzed', $results);
    $this->assertArrayHasKey('analysis_results', $results);
    $this->assertEquals(1, $results['files_analyzed']);
    $this->assertArrayHasKey($test_file, $results['analysis_results']);
  }

  /**
   * Tests getting upgrade recommendations.
   *
   * @covers ::getRecommendations
   */
  public function testGetRecommendations() {
    // Mock project info
    $this->moduleHandler->getModuleList()->willReturn([]);
    
    // Test getting recommendations
    $recommendations = $this->projectAnalyzer->getRecommendations();

    $this->assertIsArray($recommendations);
    $this->assertArrayHasKey('type', $recommendations[0]);
    $this->assertArrayHasKey('priority', $recommendations[0]);
    $this->assertArrayHasKey('message', $recommendations[0]);
  }

  /**
   * Creates a mock Extension object.
   *
   * @param string $name
   *   The module name.
   * @param array $info
   *   The module info.
   *
   * @return \Drupal\Core\Extension\Extension
   *   A mock Extension object.
   */
  protected function getMockExtension($name, array $info) {
    $extension = new \stdClass();
    $extension->info = $info;
    $extension->name = $name;
    return $extension;
  }

}
