<?php

namespace Drupal\ai_upgrade_assistant\Cache;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\QueueInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Cache warmup helper for AI Upgrade Assistant.
 *
 * This service helps maintain optimal performance by:
 * - Pre-warming caches for commonly accessed patterns
 * - Managing cache tags for upgrade patterns
 * - Implementing intelligent cache invalidation
 * - Collecting cache statistics for optimization
 */
class AiUpgradeWarmupHelper {

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The queue factory.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected $queueFactory;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The warmup queue.
   *
   * @var \Drupal\Core\Queue\QueueInterface
   */
  protected $warmupQueue;

  /**
   * Cache bin for upgrade patterns.
   *
   * @var string
   */
  const PATTERN_CACHE_BIN = 'ai_upgrade_patterns';

  /**
   * Cache bin for analysis results.
   *
   * @var string
   */
  const ANALYSIS_CACHE_BIN = 'ai_upgrade_analysis';

  /**
   * Constructs a new AiUpgradeWarmupHelper.
   *
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Queue\QueueFactory $queue_factory
   *   The queue factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(
    CacheBackendInterface $cache,
    StateInterface $state,
    ConfigFactoryInterface $config_factory,
    QueueFactory $queue_factory,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->cache = $cache;
    $this->state = $state;
    $this->configFactory = $config_factory;
    $this->queueFactory = $queue_factory;
    $this->loggerFactory = $logger_factory->get('ai_upgrade_assistant');
    $this->warmupQueue = $queue_factory->get('ai_upgrade_warmup');
  }

  /**
   * Warms up caches for commonly used patterns and configurations.
   *
   * @param bool $force
   *   Force a cache warmup even if not needed.
   */
  public function warmupCaches($force = FALSE) {
    $last_warmup = $this->state->get('ai_upgrade_assistant.last_cache_warmup', 0);
    $config = $this->configFactory->get('ai_upgrade_assistant.settings');
    $warmup_interval = $config->get('cache_warmup_interval') ?: 3600;

    // Check if warmup is needed
    if (!$force && ($last_warmup > (time() - $warmup_interval))) {
      return;
    }

    // Queue patterns for warmup
    $patterns = $this->getCommonPatterns();
    foreach ($patterns as $pattern) {
      $this->warmupQueue->createItem([
        'type' => 'pattern',
        'id' => $pattern['id'],
        'data' => $pattern,
      ]);
    }

    // Queue analysis results
    $analyses = $this->getRecentAnalyses();
    foreach ($analyses as $analysis) {
      $this->warmupQueue->createItem([
        'type' => 'analysis',
        'id' => $analysis['id'],
        'data' => $analysis,
      ]);
    }

    $this->state->set('ai_upgrade_assistant.last_cache_warmup', time());
    $this->loggerFactory->info('Cache warmup queued for @count items', [
      '@count' => count($patterns) + count($analyses),
    ]);
  }

  /**
   * Gets commonly accessed upgrade patterns.
   *
   * @return array
   *   Array of patterns to warm up.
   */
  protected function getCommonPatterns() {
    $patterns = [];
    $cached = $this->state->get('ai_upgrade_assistant.common_patterns', []);

    if (!empty($cached) && ($cached['timestamp'] > (time() - 86400))) {
      return $cached['patterns'];
    }

    // Analyze pattern usage and collect most common ones
    // Implementation will vary based on tracking mechanism
    
    $this->state->set('ai_upgrade_assistant.common_patterns', [
      'patterns' => $patterns,
      'timestamp' => time(),
    ]);

    return $patterns;
  }

  /**
   * Gets recent analysis results that should be cached.
   *
   * @return array
   *   Array of recent analyses.
   */
  protected function getRecentAnalyses() {
    // Implementation will depend on how analyses are stored
    return [];
  }

  /**
   * Collects cache statistics for optimization.
   *
   * @return array
   *   Array of cache statistics.
   */
  public function collectStatistics() {
    $stats = [
      'pattern_cache_hits' => $this->state->get('ai_upgrade_assistant.pattern_cache_hits', 0),
      'pattern_cache_misses' => $this->state->get('ai_upgrade_assistant.pattern_cache_misses', 0),
      'analysis_cache_hits' => $this->state->get('ai_upgrade_assistant.analysis_cache_hits', 0),
      'analysis_cache_misses' => $this->state->get('ai_upgrade_assistant.analysis_cache_misses', 0),
      'warmup_efficiency' => $this->calculateWarmupEfficiency(),
    ];

    $this->loggerFactory->debug('Cache statistics collected: @stats', [
      '@stats' => print_r($stats, TRUE),
    ]);

    return $stats;
  }

  /**
   * Calculates the efficiency of cache warmup.
   *
   * @return float
   *   Efficiency percentage.
   */
  protected function calculateWarmupEfficiency() {
    $hits = $this->state->get('ai_upgrade_assistant.pattern_cache_hits', 0);
    $misses = $this->state->get('ai_upgrade_assistant.pattern_cache_misses', 0);
    
    if (($hits + $misses) === 0) {
      return 0.0;
    }

    return round(($hits / ($hits + $misses)) * 100, 2);
  }

}
