<?php

namespace Drupal\ai_upgrade_assistant\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Batch\BatchBuilder;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Drupal\ai_upgrade_assistant\Service\BatchAnalyzer;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;

/**
 * Controller for handling analysis operations.
 */
class AnalysisController extends ControllerBase {
  use DependencySerializationTrait;

  /**
   * The batch analyzer service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\BatchAnalyzer
   */
  protected $batchAnalyzer;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * Constructs a new AnalysisController object.
   *
   * @param \Drupal\ai_upgrade_assistant\Service\BatchAnalyzer $batch_analyzer
   *   The batch analyzer service.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   */
  public function __construct(
    BatchAnalyzer $batch_analyzer,
    StateInterface $state,
    RendererInterface $renderer,
    DateFormatterInterface $date_formatter
  ) {
    $this->batchAnalyzer = $batch_analyzer;
    $this->state = $state;
    $this->renderer = $renderer;
    $this->dateFormatter = $date_formatter;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('ai_upgrade_assistant.batch_analyzer'),
      $container->get('state'),
      $container->get('renderer'),
      $container->get('date.formatter')
    );
  }

  /**
   * Starts the analysis process.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse|\Symfony\Component\HttpFoundation\RedirectResponse
   *   JSON response with batch token or redirect to batch processing.
   */
  public function startAnalysis() {
    // Clear any existing analysis data
    $this->state->delete('ai_upgrade_assistant.analysis_results');
    
    // Get list of enabled modules
    $modules = array_keys($this->moduleHandler()->getModuleList());
    
    // Create and set the batch
    $batch = $this->batchAnalyzer->createBatch($modules);
    
    // Set the batch
    batch_set($batch);
    
    // Process the batch
    if ($this->isAjax()) {
      $batch =& batch_get();
      $batch['progressive'] = TRUE;
      $response = [
        'status' => 'success',
        'message' => $this->t('Analysis started'),
        'batch_token' => $batch['token'],
      ];
      return new JsonResponse($response);
    }
    
    return batch_process('/admin/reports/ai-upgrade-assistant/analysis');
  }

  /**
   * Checks if the current request is an AJAX request.
   *
   * @return bool
   *   TRUE if the request is an AJAX request, FALSE otherwise.
   */
  protected function isAjax() {
    return \Drupal::request()->isXmlHttpRequest();
  }

  /**
   * Batch finished callback.
   */
  public function analysisFinished($success, $results, $operations) {
    if ($success) {
      $this->messenger()->addStatus($this->t('Module analysis completed successfully.'));
    }
    else {
      $this->messenger()->addError($this->t('There was an error during the analysis process.'));
    }
  }

  /**
   * Gets the current analysis progress.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with progress information.
   */
  public function getProgress() {
    $batch = batch_get();
    $results = $this->state->get('ai_upgrade_assistant.analysis_results', []);
    
    if (!$batch) {
      return new JsonResponse([
        'status' => 'complete',
        'current' => 100,
        'total' => 100,
        'message' => $this->t('Analysis complete'),
      ]);
    }

    $current = $batch['sets'][$batch['current_set']]['current'];
    $total = $batch['sets'][$batch['current_set']]['total'];
    
    return new JsonResponse([
      'status' => 'in_progress',
      'current' => $current,
      'total' => $total,
      'message' => $batch['sets'][$batch['current_set']]['message'],
      'terminal_output' => $this->getLatestTerminalOutput(),
    ]);
  }

  /**
   * Gets the latest recommendations.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with rendered recommendations.
   */
  public function getRecommendations() {
    $results = $this->state->get('ai_upgrade_assistant.analysis_results', []);
    
    $build = [
      '#theme' => 'upgrade_recommendations',
      '#recommendations' => $this->processResults($results),
    ];

    return new JsonResponse([
      'content' => $this->renderer->render($build),
    ]);
  }

  /**
   * Gets the code diff view.
   *
   * @return array
   *   Render array for the diff view.
   */
  public function getDiffView($file_path) {
    $results = $this->state->get('ai_upgrade_assistant.analysis_results', []);
    $file_results = $results['files'][$file_path] ?? [];
    
    return [
      '#theme' => 'code_diff_view',
      '#file_path' => $file_path,
      '#analysis' => $file_results,
      '#attached' => [
        'library' => ['ai_upgrade_assistant/diff_view'],
      ],
    ];
  }

  /**
   * Gets the latest terminal output.
   *
   * @return string
   *   The latest terminal output.
   */
  protected function getLatestTerminalOutput() {
    return $this->state->get('ai_upgrade_assistant.terminal_output', '');
  }

  /**
   * Processes analysis results into recommendations.
   *
   * @param array $results
   *   Raw analysis results.
   *
   * @return array
   *   Processed recommendations.
   */
  protected function processResults(array $results) {
    $recommendations = [];

    // Process core analysis
    if (!empty($results['core'])) {
      if (!$results['core']['compatible']) {
        $recommendations[] = [
          'type' => 'core_upgrade',
          'priority' => 'high',
          'message' => $this->t('Upgrade to Drupal 10 required. Current version: @version', 
            ['@version' => $results['core']['version']]),
        ];
      }
    }

    // Process module analysis
    if (!empty($results['modules'])) {
      foreach ($results['modules'] as $name => $module) {
        if (!$module['compatible']) {
          $recommendations[] = [
            'type' => 'module_compatibility',
            'priority' => 'medium',
            'message' => $this->t('Module @name needs to be updated for Drupal 10 compatibility',
              ['@name' => $name]),
            'details' => $module['issues'],
          ];
        }
      }
    }

    // Process file analysis
    if (!empty($results['files'])) {
      foreach ($results['files'] as $file_path => $analysis) {
        if (!empty($analysis['issues'])) {
          foreach ($analysis['issues'] as $issue) {
            $recommendations[] = [
              'type' => $issue['type'],
              'priority' => $issue['priority'],
              'message' => $issue['description'],
              'actions' => [
                [
                  'label' => $this->t('View changes'),
                  'url' => "admin/reports/upgrade-assistant/diff/" . urlencode($file_path),
                ],
              ],
              'code_example' => $issue['code_example'] ?? NULL,
            ];
          }
        }
      }
    }

    return $recommendations;
  }

  /**
   * Analyzes a specific module.
   *
   * @param string $module
   *   The machine name of the module to analyze.
   *
   * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
   *   A render array or redirect response.
   */
  public function analyzeModule($module) {
    // Create batch with single module
    $batch = $this->batchAnalyzer->createBatch([$module]);
    batch_set($batch);

    if (PHP_SAPI === 'cli') {
      drush_backend_batch_process();
      return [
        '#markup' => $this->t('Module analysis complete.'),
      ];
    }

    return batch_process("/admin/reports/upgrade-assistant/module/$module");
  }

  /**
   * Displays detailed analysis results for a module.
   *
   * @param string $module
   *   The machine name of the module.
   *
   * @return array
   *   A render array for the module details page.
   */
  public function moduleDetails($module) {
    $analysis_results = $this->state->get("ai_upgrade_assistant.analysis_results.$module", []);
    
    return [
      '#theme' => 'analysis_report',
      '#module' => $module,
      '#results' => $analysis_results,
      '#attached' => [
        'library' => ['ai_upgrade_assistant/analysis_report'],
      ],
    ];
  }

  /**
   * Displays the analysis report.
   *
   * @return array
   *   A render array representing the analysis report.
   */
  public function displayReport() {
    $results = $this->state->get('ai_upgrade_assistant.analysis_results', []);
    
    // Process the results
    $summary = $this->calculateSummary($results);
    $moduleAnalysis = $this->processModuleAnalysis($results);
    $securityIssues = $this->processSecurityIssues($results);
    $recommendations = $this->processResults($results);

    return [
      '#theme' => 'analysis-report',
      '#data' => [
        'generated_on' => $this->dateFormatter->format(time(), 'custom', 'Y-m-d H:i:s T'),
        'drupal_version' => \Drupal::VERSION,
        'summary' => $summary,
        'module_analysis' => $moduleAnalysis,
        'security_issues' => $securityIssues,
        'recommendations' => $recommendations,
      ],
      '#attached' => [
        'library' => ['ai_upgrade_assistant/analysis_report'],
      ],
    ];
  }

  /**
   * Calculates summary statistics from analysis results.
   */
  protected function calculateSummary($results) {
    $summary = [
      'total_files' => 0,
      'total_issues' => 0,
      'critical_issues' => 0,
      'warnings' => 0,
      'suggestions' => 0,
    ];

    if (!empty($results['files'])) {
      $summary['total_files'] = count($results['files']);
      
      foreach ($results['files'] as $file) {
        if (!empty($file['issues'])) {
          foreach ($file['issues'] as $issue) {
            $summary['total_issues']++;
            switch ($issue['severity']) {
              case 'critical':
                $summary['critical_issues']++;
                break;
              case 'warning':
                $summary['warnings']++;
                break;
              case 'suggestion':
                $summary['suggestions']++;
                break;
            }
          }
        }
      }
    }

    return $summary;
  }

  /**
   * Processes module analysis results.
   */
  protected function processModuleAnalysis($results) {
    $moduleAnalysis = [];

    if (!empty($results['modules'])) {
      foreach ($results['modules'] as $name => $module) {
        $moduleAnalysis[$name] = [
          'name' => $name,
          'version' => $module['version'] ?? 'Unknown',
          'status' => $module['compatible'] ? 'Compatible' : 'Needs Update',
          'issues' => $module['issues'] ?? [],
          'dependencies' => $module['dependencies'] ?? [],
          'security_issues' => $module['security_issues'] ?? [],
          'available_updates' => $module['available_updates'] ?? [],
        ];
      }
    }

    return $moduleAnalysis;
  }

  /**
   * Processes security issues from results.
   */
  protected function processSecurityIssues($results) {
    $securityIssues = [];

    if (!empty($results['security'])) {
      foreach ($results['security'] as $module => $issues) {
        foreach ($issues as $issue) {
          $securityIssues[] = [
            'module' => $module,
            'title' => $issue['title'],
            'severity' => $issue['severity'],
            'description' => $issue['description'],
            'solution' => $issue['solution'] ?? '',
            'advisory_link' => $issue['link'] ?? '',
          ];
        }
      }
    }

    return $securityIssues;
  }
}
