<?php

namespace Drupal\ai_upgrade_assistant\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Render\Markup;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\ai_upgrade_assistant\Service\ModuleAnalyzerService;
use Drupal\ai_upgrade_assistant\Service\EnvironmentCheckerService;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\State\StateInterface;

/**
 * Controller for the AI Upgrade Assistant status page.
 */
class StatusController extends ControllerBase {

  /**
   * The module analyzer service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\ModuleAnalyzerService
   */
  protected $moduleAnalyzer;

  /**
   * The environment checker service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\EnvironmentCheckerService
   */
  protected $environmentChecker;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * Constructs a StatusController object.
   */
  public function __construct(
    ModuleAnalyzerService $module_analyzer,
    EnvironmentCheckerService $environment_checker,
    ModuleHandlerInterface $module_handler,
    ConfigFactoryInterface $config_factory,
    StateInterface $state
  ) {
    $this->moduleAnalyzer = $module_analyzer;
    $this->environmentChecker = $environment_checker;
    $this->moduleHandler = $module_handler;
    $this->configFactory = $config_factory;
    $this->state = $state;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('ai_upgrade_assistant.module_analyzer'),
      $container->get('ai_upgrade_assistant.environment_checker'),
      $container->get('module_handler'),
      $container->get('config.factory'),
      $container->get('state')
    );
  }

  /**
   * Displays the status page.
   *
   * @return array
   *   A render array representing the status page.
   */
  public function statusPage() {
    // Get environment checks.
    $environment_checks = [
      [
        '#type' => 'container',
        'status' => [
          '#markup' => version_compare(PHP_VERSION, '8.1.0', '>=') ? 'success' : 'error',
        ],
        'message' => [
          '#markup' => $this->t('PHP Version: @version', ['@version' => PHP_VERSION]),
        ],
      ],
      [
        '#type' => 'container',
        'status' => [
          '#markup' => $this->moduleHandler->moduleExists('update') ? 'success' : 'error',
        ],
        'message' => [
          '#markup' => $this->t('Update module status'),
        ],
      ],
      [
        '#type' => 'container',
        'status' => [
          '#markup' => !empty($this->state->get('ai_upgrade_assistant.huggingface_api_key')) ? 'success' : 'error',
        ],
        'message' => [
          '#markup' => $this->t('HuggingFace API key configuration'),
        ],
      ],
    ];

    // Get all installed modules
    $modules = $this->moduleHandler->getModuleList();
    $module_groups = [
      'Compatible' => [],
      'Needs_Update' => [],
      'In_Progress' => [],
      'Error' => [],
      'Unknown' => [],
    ];

    foreach ($modules as $name => $module) {
      $info = $this->moduleHandler->getModule($name)->info;
      $status = $this->moduleAnalyzer->getModuleStatus($name);
      
      $module_data = [
        '#type' => 'container',
        'name' => [
          '#markup' => $name,
        ],
        'version' => [
          '#markup' => $info['version'] ?? '8.x-1.x',
        ],
        'status' => [
          '#markup' => $status['status'] ?? 'unknown',
        ],
      ];

      if ($status['status'] === 'in_progress' && isset($status['progress'])) {
        $module_data['progress'] = [
          '#markup' => $status['progress'],
        ];
      }

      $group = $status['status'] ?? 'Unknown';
      $module_groups[ucfirst($group)][] = $module_data;
    }

    // Calculate overall progress
    $total_modules = count($modules);
    $completed_modules = count($module_groups['Compatible'] ?? []);
    $progress_percentage = $total_modules > 0 ? round(($completed_modules / $total_modules) * 100) : 0;

    // Build the render array
    return [
      '#theme' => 'ai_upgrade_assistant_status',
      '#attached' => [
        'library' => ['ai_upgrade_assistant/status_page'],
      ],
      '#environment' => [
        '#type' => 'container',
        'items' => $environment_checks,
      ],
      '#modules' => [
        '#type' => 'container',
        'groups' => $module_groups,
      ],
      '#progress' => [
        '#type' => 'container',
        'data' => [
          '#markup' => $progress_percentage,
        ],
      ],
      '#cache' => [
        'max-age' => 0,
      ],
    ];
  }

}
