<?php

namespace Drupal\ai_upgrade_assistant\Service;

/**
 * Service for generating AI prompts for code analysis.
 */
class AiPromptGenerator {

  /**
   * Generates a prompt for analyzing deprecated code.
   *
   * @param array $deprecated_items
   *   Array of deprecated items found in the code.
   * @param string $file_path
   *   The file path being analyzed.
   * @param string $module_name
   *   The module name.
   *
   * @return string
   *   The generated prompt.
   */
  public function generateDeprecationAnalysisPrompt(array $deprecated_items, string $file_path, string $module_name) {
    $prompt = "Analyze the following deprecated code items found in {$file_path} of the {$module_name} module:\n\n";

    foreach ($deprecated_items as $item) {
      $prompt .= "- {$item['type']}: {$item['name']}\n";
      $prompt .= "  Line: {$item['line']}\n";
      $prompt .= "  Replacement: {$item['replacement']}\n";
      $prompt .= "  Deprecated since: {$item['version']}\n\n";
    }

    $prompt .= "\nPlease provide:\n";
    $prompt .= "1. A detailed explanation of why each item is deprecated\n";
    $prompt .= "2. Step-by-step instructions for updating each deprecated item\n";
    $prompt .= "3. Code examples showing the old and new implementations\n";
    $prompt .= "4. Any potential side effects or considerations when updating\n";
    $prompt .= "5. Additional recommendations for improving the code\n";

    return $prompt;
  }

  /**
   * Generates a prompt for analyzing hook implementations.
   *
   * @param array $hooks
   *   Array of hooks found in the code.
   * @param string $module_name
   *   The module name.
   *
   * @return string
   *   The generated prompt.
   */
  public function generateHookAnalysisPrompt(array $hooks, string $module_name) {
    $prompt = "Analyze the following hook implementations in the {$module_name} module:\n\n";

    foreach ($hooks as $hook) {
      $prompt .= "- Hook: {$hook['name']}\n";
      $prompt .= "  File: {$hook['file']}\n";
      $prompt .= "  Line: {$hook['line']}\n";
      $prompt .= "  Parameters: " . implode(', ', $hook['params']) . "\n\n";
    }

    $prompt .= "\nPlease provide:\n";
    $prompt .= "1. An analysis of each hook's purpose and typical use cases\n";
    $prompt .= "2. Recommendations for improving the implementations\n";
    $prompt .= "3. Potential performance implications\n";
    $prompt .= "4. Security considerations\n";
    $prompt .= "5. Best practices for hook implementations\n";
    $prompt .= "6. Suggestions for alternative approaches if applicable\n";

    return $prompt;
  }

  /**
   * Generates a prompt for analyzing class usage.
   *
   * @param array $class_usage
   *   Array of class usage information.
   *
   * @return string
   *   The generated prompt.
   */
  public function generateClassAnalysisPrompt(array $class_usage) {
    $prompt = "Analyze the following class usage patterns:\n\n";

    foreach ($class_usage as $usage) {
      $prompt .= "- Type: {$usage['type']}\n";
      if (isset($usage['class'])) {
        $prompt .= "  Class: {$usage['class']}\n";
      }
      if (isset($usage['interface'])) {
        $prompt .= "  Interface: {$usage['interface']}\n";
      }
      $prompt .= "  File: {$usage['file']}\n";
      $prompt .= "  Line: {$usage['line']}\n\n";
    }

    $prompt .= "\nPlease provide:\n";
    $prompt .= "1. An analysis of the class architecture and design patterns\n";
    $prompt .= "2. Recommendations for improving class organization\n";
    $prompt .= "3. Potential dependency injection improvements\n";
    $prompt .= "4. Interface compliance analysis\n";
    $prompt .= "5. Suggestions for better abstraction if applicable\n";
    $prompt .= "6. Performance considerations\n";

    return $prompt;
  }

  /**
   * Generates a prompt for patch generation.
   *
   * @param array $analysis_results
   *   The complete analysis results.
   * @param string $module_name
   *   The module name.
   *
   * @return string
   *   The generated prompt.
   */
  public function generatePatchPrompt(array $analysis_results, string $module_name) {
    $prompt = "Generate a patch for the {$module_name} module based on the following analysis:\n\n";

    if (!empty($analysis_results['deprecated_items'])) {
      $prompt .= "Deprecated Items to Update:\n";
      foreach ($analysis_results['deprecated_items'] as $item) {
        $prompt .= "- {$item['type']}: {$item['name']} -> {$item['replacement']}\n";
        $prompt .= "  File: {$item['file']}, Line: {$item['line']}\n\n";
      }
    }

    $prompt .= "\nRequirements for the patch:\n";
    $prompt .= "1. Follow Drupal coding standards\n";
    $prompt .= "2. Maintain backward compatibility where possible\n";
    $prompt .= "3. Include appropriate documentation updates\n";
    $prompt .= "4. Add upgrade path notes if needed\n";
    $prompt .= "5. Consider performance implications\n";
    $prompt .= "6. Follow security best practices\n";

    return $prompt;
  }

  /**
   * Generates a prompt for analyzing potential security issues.
   *
   * @param array $analysis_results
   *   The complete analysis results.
   *
   * @return string
   *   The generated prompt.
   */
  public function generateSecurityAnalysisPrompt(array $analysis_results) {
    $prompt = "Perform a security analysis of the following code patterns:\n\n";

    // Analyze class usage for potential security issues
    if (!empty($analysis_results['analysis']['class_usage'])) {
      $prompt .= "Class Usage Patterns:\n";
      foreach ($analysis_results['analysis']['class_usage'] as $usage) {
        $prompt .= "- {$usage['type']}: " . ($usage['class'] ?? $usage['interface']) . "\n";
        $prompt .= "  File: {$usage['file']}, Line: {$usage['line']}\n\n";
      }
    }

    // Analyze hooks for security considerations
    if (!empty($analysis_results['analysis']['hooks'])) {
      $prompt .= "Hook Implementations:\n";
      foreach ($analysis_results['analysis']['hooks'] as $hook) {
        $prompt .= "- {$hook['name']}\n";
        $prompt .= "  Parameters: " . implode(', ', $hook['params']) . "\n\n";
      }
    }

    $prompt .= "\nPlease analyze for:\n";
    $prompt .= "1. Potential SQL injection vulnerabilities\n";
    $prompt .= "2. Cross-site scripting (XSS) risks\n";
    $prompt .= "3. Access control issues\n";
    $prompt .= "4. Input validation concerns\n";
    $prompt .= "5. Authentication and authorization gaps\n";
    $prompt .= "6. Data sanitization requirements\n";
    $prompt .= "7. Security implications of deprecated code usage\n";

    return $prompt;
  }

}
