<?php

namespace Drupal\ai_upgrade_assistant\Service;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Cache\CacheTagsInvalidatorInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\QueueInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Service for warming and collecting AI upgrade pattern caches.
 *
 * This service ensures that commonly used upgrade patterns are pre-cached
 * and that successful patterns from the community are collected and shared.
 */
class CacheWarmer implements CacheWarmerInterface {

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The cache tags invalidator.
   *
   * @var \Drupal\Core\Cache\CacheTagsInvalidatorInterface
   */
  protected $cacheTagsInvalidator;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The queue for processing pattern collection.
   *
   * @var \Drupal\Core\Queue\QueueInterface
   */
  protected $queue;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * The community learning service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\CommunityLearningService
   */
  protected $communityLearning;

  /**
   * Constructs a new CacheWarmer object.
   */
  public function __construct(
    CacheBackendInterface $cache,
    CacheTagsInvalidatorInterface $cacheTagsInvalidator,
    ConfigFactoryInterface $configFactory,
    StateInterface $state,
    QueueFactory $queueFactory,
    LoggerChannelFactoryInterface $loggerFactory,
    CommunityLearningService $communityLearning
  ) {
    $this->cache = $cache;
    $this->cacheTagsInvalidator = $cacheTagsInvalidator;
    $this->configFactory = $configFactory;
    $this->state = $state;
    $this->queue = $queueFactory->get('ai_upgrade_pattern_collector');
    $this->logger = $loggerFactory->get('ai_upgrade_assistant');
    $this->communityLearning = $communityLearning;
  }

  /**
   * {@inheritdoc}
   */
  public function warmCache($moduleNames = []) {
    if (empty($moduleNames)) {
      // Get all installed modules if none specified
      $moduleNames = array_keys(\Drupal::service('module_handler')->getModuleList());
    }

    $config = $this->configFactory->get('ai_upgrade_assistant.settings');
    $batchSize = $config->get('cache_warm_batch_size') ?? 50;
    $patterns = [];

    foreach ($moduleNames as $moduleName) {
      // Queue pattern collection for this module
      $this->queue->createItem([
        'module_name' => $moduleName,
        'operation' => 'collect_patterns',
      ]);

      // Pre-warm with known successful patterns
      $knownPatterns = $this->communityLearning->getSuccessfulPatterns($moduleName);
      if (!empty($knownPatterns)) {
        $patterns[$moduleName] = $knownPatterns;
      }

      if (count($patterns) >= $batchSize) {
        $this->cachePatterns($patterns);
        $patterns = [];
      }
    }

    // Cache any remaining patterns
    if (!empty($patterns)) {
      $this->cachePatterns($patterns);
    }

    $this->state->set('ai_upgrade_assistant.last_cache_warm', REQUEST_TIME);
    $this->logger->info('Cache warming completed for @count modules', [
      '@count' => count($moduleNames),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function collectPatterns($moduleNames = []) {
    if (empty($moduleNames)) {
      $moduleNames = array_keys(\Drupal::service('module_handler')->getModuleList());
    }

    foreach ($moduleNames as $moduleName) {
      $this->queue->createItem([
        'module_name' => $moduleName,
        'operation' => 'analyze_patterns',
      ]);
    }

    $this->logger->info('Pattern collection queued for @count modules', [
      '@count' => count($moduleNames),
    ]);
  }

  /**
   * Cache a batch of patterns.
   *
   * @param array $patterns
   *   Array of patterns to cache, keyed by module name.
   */
  protected function cachePatterns(array $patterns) {
    foreach ($patterns as $moduleName => $modulePatterns) {
      $cacheId = "ai_upgrade_assistant:patterns:$moduleName";
      $this->cache->set(
        $cacheId,
        $modulePatterns,
        CacheBackendInterface::CACHE_PERMANENT,
        ['ai_upgrade_assistant:patterns', "ai_upgrade_assistant:module:$moduleName"]
      );
    }
  }

  /**
   * {@inheritdoc}
   */
  public function invalidateModuleCache($moduleName) {
    $this->cacheTagsInvalidator->invalidateTags([
      'ai_upgrade_assistant:patterns',
      "ai_upgrade_assistant:module:$moduleName",
    ]);
  }

}
