<?php

namespace Drupal\ai_upgrade_assistant\Service;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use PhpParser\Node;
use PhpParser\NodeTraverser;
use PhpParser\NodeVisitorAbstract;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;

/**
 * Service for analyzing Drupal projects for upgrade compatibility.
 */
class ProjectAnalyzer {
  use DependencySerializationTrait;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The module extension list.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected $moduleExtensionList;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The HuggingFace service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\HuggingFaceService
   */
  protected $huggingFace;

  /**
   * The PHP parser service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\PhpParserService
   */
  protected $phpParser;

  /**
   * The AI model manager.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\AiModelManager
   */
  protected $aiModelManager;

  /**
   * The community learning service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\CommunityLearningService
   */
  protected $communityLearning;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The analysis tracker service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\AnalysisTracker
   */
  protected $analysisTracker;

  /**
   * Constructs a new ProjectAnalyzer object.
   */
  public function __construct(
    ModuleHandlerInterface $moduleHandler,
    ModuleExtensionList $moduleExtensionList,
    ConfigFactoryInterface $configFactory,
    FileSystemInterface $fileSystem,
    LoggerChannelFactoryInterface $loggerFactory,
    HuggingFaceService $huggingFace,
    PhpParserService $phpParser,
    AiModelManager $aiModelManager,
    CommunityLearningService $communityLearning,
    ClientInterface $httpClient,
    AnalysisTracker $analysisTracker
  ) {
    $this->moduleHandler = $moduleHandler;
    $this->moduleExtensionList = $moduleExtensionList;
    $this->configFactory = $configFactory;
    $this->fileSystem = $fileSystem;
    $this->loggerFactory = $loggerFactory;
    $this->huggingFace = $huggingFace;
    $this->phpParser = $phpParser;
    $this->aiModelManager = $aiModelManager;
    $this->communityLearning = $communityLearning;
    $this->httpClient = $httpClient;
    $this->analysisTracker = $analysisTracker;
  }

  /**
   * Creates a new instance of the service.
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('module_handler'),
      $container->get('extension.list.module'),
      $container->get('config.factory'),
      $container->get('file_system'),
      $container->get('logger.factory'),
      $container->get('ai_upgrade_assistant.huggingface'),
      $container->get('ai_upgrade_assistant.php_parser'),
      $container->get('ai_upgrade_assistant.ai_model_manager'),
      $container->get('ai_upgrade_assistant.community_learning'),
      $container->get('http_client'),
      $container->get('ai_upgrade_assistant.analysis_tracker')
    );
  }

  /**
   * Analyzes code for potential upgrade issues.
   *
   * @param string $module
   *   The name of the module to analyze.
   * @param string $path
   *   Optional path to analyze. If not provided, will use module path.
   *
   * @return array
   *   Analysis results keyed by file path.
   */
  public function analyzeCode($module, $path = NULL) {
    $results = [];

    try {
      // Get module info if path not provided
      if (!$path) {
        $module_data = $this->moduleExtensionList->get($module);
        if (!$module_data) {
          throw new \Exception('Module not found: ' . $module);
        }
        $path = $module_data->getPath();
      }

      // Ensure path exists
      if (!file_exists($path)) {
        throw new \Exception('Path does not exist: ' . $path);
      }

      // Find all PHP files
      $files = is_dir($path) ? $this->findPhpFiles($path) : [$path];

      foreach ($files as $file) {
        try {
          if (!file_exists($file)) {
            $this->loggerFactory->get('ai_upgrade_assistant')->warning('File does not exist: @file', ['@file' => $file]);
            continue;
          }

          $code_string = file_get_contents($file);
          if ($code_string === FALSE) {
            $this->loggerFactory->get('ai_upgrade_assistant')->warning('Could not read file: @file', ['@file' => $file]);
            continue;
          }

          $ast = $this->phpParser->parseFile($file);
          if (!$ast) {
            $this->loggerFactory->get('ai_upgrade_assistant')->warning('Could not parse file @file', ['@file' => $file]);
            continue;
          }

          $file_config = $this->buildCodeContext($ast, $file);
          $patterns = $this->communityLearning->findSimilarPatterns($file);
          
          if ($patterns) {
            $file_config['patterns'] = array_map(function ($pattern) {
              return is_object($pattern) && isset($pattern->pattern_data) 
                ? unserialize($pattern->pattern_data) 
                : [];
            }, $patterns);
          }

          // Try AI analysis first, fall back to basic analysis if not available
          try {
            $results[$file] = $this->huggingFace->analyzeCode($code_string, $file_config);
          }
          catch (\Exception $e) {
            if (strpos($e->getMessage(), 'subscription') !== FALSE) {
              // Perform basic analysis without AI
              $results[$file] = $this->performBasicAnalysis($ast, $file);
            }
            else {
              throw $e;
            }
          }
        }
        catch (\Exception $e) {
          $this->loggerFactory->get('ai_upgrade_assistant')->error('Error analyzing file @file: @error', [
            '@file' => $file,
            '@error' => $e->getMessage(),
          ]);
          $results[$file] = [
            'status' => 'error',
            'message' => $e->getMessage(),
          ];
        }
      }
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('ai_upgrade_assistant')->error('Error analyzing module @module: @error', [
        '@module' => $module,
        '@error' => $e->getMessage(),
      ]);
      return [
        'status' => 'error',
        'message' => $e->getMessage(),
      ];
    }

    return $results;
  }

  /**
   * Analyzes security issues for a module.
   */
  public function analyzeSecurityIssues($module) {
    try {
      $module_data = $this->moduleExtensionList->get($module);
      if (!$module_data) {
        throw new \Exception('Module not found: ' . $module);
      }

      $advisories = $this->fetchSecurityAdvisories($module);
      $code_issues = [];
      
      $path = $module_data->getPath();
      $files = $this->findPhpFiles($path);
      
      foreach ($files as $file) {
        try {
          $ast = $this->phpParser->parseFile($file);
          if (!$ast) {
            $this->loggerFactory->get('ai_upgrade_assistant')->warning('Unable to parse file @file', ['@file' => $file]);
            continue;
          }
          
          $file_issues = $this->analyzeCodeSecurity($ast, $file);
          if (!empty($file_issues)) {
            $code_issues[$file] = $file_issues;
          }
        }
        catch (\Exception $e) {
          $this->loggerFactory->get('ai_upgrade_assistant')->error('Error analyzing file @file: @error', [
            '@file' => $file,
            '@error' => $e->getMessage(),
          ]);
        }
      }

      $risk_level = $this->calculateSecurityRiskLevel($advisories, $code_issues);

      return [
        'status' => 'analyzed',
        'risk_level' => $risk_level,
        'advisories' => $advisories,
        'code_issues' => $code_issues,
      ];
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('ai_upgrade_assistant')->error('Error analyzing security for module @module: @error', [
        '@module' => $module,
        '@error' => $e->getMessage(),
      ]);
      return [
        'status' => 'error',
        'message' => $e->getMessage(),
      ];
    }
  }

  /**
   * Analyzes dependencies for a module.
   *
   * @param string $module
   *   The name of the module to analyze.
   *
   * @return array
   *   An array containing dependency analysis results.
   *
   * @throws \Exception
   *   If the module is not found or other errors occur.
   */
  public function analyzeDependencies($module) {
    try {
      $module_data = $this->moduleExtensionList->get($module);
      if (!$module_data) {
        throw new \Exception('Module not found: ' . $module);
      }

      $info = $module_data->info;
      $results = [
        'core' => [],
        'contrib' => [],
        'custom' => [],
        'themes' => [],
        'libraries' => [],
        'status' => 'success',
      ];

      // Check core version compatibility
      if (isset($info['core_version_requirement'])) {
        $results['core']['version_requirement'] = $info['core_version_requirement'];
      }
      elseif (isset($info['core'])) {
        $results['core']['version_requirement'] = $info['core'];
      }

      // Analyze module dependencies
      if (isset($info['dependencies'])) {
        foreach ($info['dependencies'] as $dependency) {
          $dependency_parts = explode(':', $dependency);
          $dependency_name = end($dependency_parts);
          $dependency_type = count($dependency_parts) > 1 ? $dependency_parts[0] : 'module';

          try {
            $dep_info = $this->moduleExtensionList->get($dependency_name);
            $is_custom = strpos($dep_info->getPath(), 'modules/custom') !== FALSE;
            
            $dep_data = [
              'name' => $dependency_name,
              'type' => $dependency_type,
              'version' => isset($dep_info->info['version']) ? $dep_info->info['version'] : NULL,
              'status' => $this->moduleHandler->moduleExists($dependency_name) ? 'enabled' : 'disabled',
              'path' => $dep_info->getPath(),
            ];

            if ($is_custom) {
              $results['custom'][$dependency_name] = $dep_data;
            }
            else {
              $results['contrib'][$dependency_name] = $dep_data;
            }
          }
          catch (\Exception $e) {
            $results['missing'][$dependency_name] = [
              'name' => $dependency_name,
              'type' => $dependency_type,
              'error' => $e->getMessage(),
            ];
          }
        }
      }

      // Analyze theme dependencies
      if (isset($info['themes'])) {
        foreach ($info['themes'] as $theme) {
          $results['themes'][$theme] = [
            'name' => $theme,
            'status' => \Drupal::service('theme_handler')->themeExists($theme) ? 'installed' : 'missing',
          ];
        }
      }

      // Analyze library dependencies
      if (isset($info['libraries'])) {
        foreach ($info['libraries'] as $library) {
          $library_exists = \Drupal::service('library.discovery')->getLibraryByName($module, $library);
          $results['libraries'][$library] = [
            'name' => $library,
            'status' => $library_exists ? 'found' : 'missing',
          ];
        }
      }

      return $results;
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('ai_upgrade_assistant')->error('Error analyzing dependencies for module @module: @error', [
        '@module' => $module,
        '@error' => $e->getMessage(),
      ]);
      
      return [
        'status' => 'error',
        'message' => $e->getMessage(),
      ];
    }
  }

  /**
   * Fetches security advisories for a module.
   *
   * @param string $module_name
   *   The name of the module to check.
   *
   * @return array
   *   Array of security advisories.
   */
  protected function fetchSecurityAdvisories($module_name) {
    $advisories = [];
    $logger = $this->loggerFactory->get('ai_upgrade_assistant');

    try {
      // Try the security advisory feed first as it's more reliable
      $saUrl = "https://www.drupal.org/api/sa/{$module_name}/drupal";
      $logger->debug('Fetching security advisories from @url', ['@url' => $saUrl]);

      $saResponse = $this->httpClient->request('GET', $saUrl, [
        'headers' => ['Accept' => 'application/json'],
        'http_errors' => false,
      ]);

      if ($saResponse->getStatusCode() == 200) {
        $saData = json_decode($saResponse->getBody(), TRUE);
        if (!empty($saData)) {
          foreach ($saData as $advisory) {
            $advisories[] = [
              'title' => $advisory['title'] ?? 'Unknown',
              'link' => $advisory['link'] ?? "https://www.drupal.org/sa/{$advisory['sa_id']}",
              'severity' => $advisory['risk_level'] ?? 'Unknown',
              'version' => $advisory['covered_versions'] ?? 'All',
              'date' => isset($advisory['created']) ? strtotime($advisory['created']) : time(),
              'description' => $advisory['description'] ?? '',
              'solution' => $advisory['solution'] ?? '',
            ];
          }
          return $advisories;
        }
      }

      // Fallback to the project releases feed
      $releaseUrl = "https://www.drupal.org/api/project/{$module_name}/releases";
      $logger->debug('Fetching security releases from @url', ['@url' => $releaseUrl]);

      $response = $this->httpClient->request('GET', $releaseUrl, [
        'headers' => ['Accept' => 'application/json'],
        'http_errors' => false,
      ]);

      if ($response->getStatusCode() == 200) {
        $data = json_decode($response->getBody(), TRUE);
        if (!empty($data)) {
          foreach ($data as $release) {
            // Only include security releases
            if (isset($release['security']['covered']) && $release['security']['covered']) {
              $advisories[] = [
                'title' => $release['title'] ?? 'Unknown',
                'link' => $release['link'] ?? '',
                'severity' => 'Security',
                'version' => $release['version'] ?? 'Unknown',
                'date' => isset($release['date']) ? strtotime($release['date']) : time(),
                'description' => $release['release_notes'] ?? '',
              ];
            }
          }
        }
      }

      // If both methods fail, try the legacy feed as last resort
      if (empty($advisories)) {
        $legacyUrl = "https://www.drupal.org/feeds/project/{$module_name}/sa.json";
        $logger->debug('Fetching from legacy feed @url', ['@url' => $legacyUrl]);

        $legacyResponse = $this->httpClient->request('GET', $legacyUrl, [
          'headers' => ['Accept' => 'application/json'],
          'http_errors' => false,
        ]);

        if ($legacyResponse->getStatusCode() == 200) {
          $legacyData = json_decode($legacyResponse->getBody(), TRUE);
          if (!empty($legacyData)) {
            foreach ($legacyData as $advisory) {
              $advisories[] = [
                'title' => $advisory['title'] ?? 'Unknown',
                'link' => $advisory['link'] ?? '',
                'severity' => $advisory['risk'] ?? 'Unknown',
                'version' => $advisory['version'] ?? 'All',
                'date' => isset($advisory['created']) ? strtotime($advisory['created']) : time(),
                'description' => $advisory['description'] ?? '',
              ];
            }
          }
        }
      }

      return $advisories;
    }
    catch (\Exception $e) {
      $logger->error(
        'Error fetching security advisories for @module: @error',
        [
          '@module' => $module_name,
          '@error' => $e->getMessage(),
        ]
      );
      return [];
    }
  }

  /**
   * Finds all PHP files in a directory.
   *
   * @param string $directory
   *   Directory to search in.
   *
   * @return array
   *   Array of file paths.
   */
  protected function findPhpFiles($directory) {
    $files = [];
    
    // Ensure we have an absolute path
    if (!$this->fileSystem->realpath($directory)) {
      // Try prepending DRUPAL_ROOT
      $drupal_root = \Drupal::root();
      $full_path = $drupal_root . '/' . $directory;
      
      if (!$this->fileSystem->realpath($full_path)) {
        $this->loggerFactory->get('ai_upgrade_assistant')->error(
          'Directory not found: @dir',
          ['@dir' => $directory]
        );
        return [];
      }
      
      $directory = $full_path;
    }

    if (!is_dir($directory)) {
      $this->loggerFactory->get('ai_upgrade_assistant')->error(
        'Not a directory: @dir',
        ['@dir' => $directory]
      );
      return [];
    }

    $iterator = new \RecursiveIteratorIterator(
      new \RecursiveDirectoryIterator($directory, \RecursiveDirectoryIterator::SKIP_DOTS)
    );

    foreach ($iterator as $file) {
      if ($file->isFile() && $file->getExtension() === 'php') {
        $files[] = $file->getPathname();
      }
    }

    return $files;
  }

  /**
   * Builds context for code analysis.
   */
  protected function buildCodeContext(array $ast, $file) {
    return [
      'module' => $this->getModuleNameFromPath($file),
      'dependencies' => $this->getDependencies($file),
      'file_type' => $this->determineFileType($file),
    ];
  }

  /**
   * Gets the module name from a file path.
   */
  protected function getModuleNameFromPath($file_path) {
    $parts = explode('/modules/', $file_path);
    if (count($parts) < 2) {
      return NULL;
    }
    
    $module_path = explode('/', $parts[1]);
    return $module_path[0];
  }

  /**
   * Determines the type of a file based on its location and content.
   */
  protected function determineFileType($file) {
    $path_parts = explode('/', $file);
    $filename = end($path_parts);
    
    if (strpos($file, '/src/Plugin/') !== FALSE) {
      return 'plugin';
    }
    elseif (strpos($file, '/src/Controller/') !== FALSE) {
      return 'controller';
    }
    elseif (strpos($file, '/src/Form/') !== FALSE) {
      return 'form';
    }
    elseif ($filename === 'module') {
      return 'module';
    }
    elseif ($filename === 'install') {
      return 'install';
    }
    
    return 'unknown';
  }

  /**
   * Gets dependencies for a file.
   */
  public function getDependencies($file) {
    $module_name = $this->getModuleNameFromPath($file);
    if (!$module_name) {
      return [];
    }

    try {
      $module = $this->moduleExtensionList->get($module_name);
      $required_by = [];
      
      // Find modules that require this module
      foreach ($this->moduleHandler->getModuleList() as $name => $extension) {
        if (isset($extension->requires[$module_name])) {
          $required_by[] = $name;
        }
      }
      
      return [
        'dependencies' => $module->info['dependencies'] ?? [],
        'required_by' => $required_by,
      ];
    }
    catch (\Exception $e) {
      return [];
    }
  }

  /**
   * Analyzes code for security issues.
   */
  protected function analyzeCodeSecurity($ast, $file) {
    $issues = [];
    
    try {
      // Add security analysis logic here
      // This is a placeholder for actual security analysis
      
      return $issues;
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('ai_upgrade_assistant')->error('Error during security analysis of @file: @error', [
        '@file' => $file,
        '@error' => $e->getMessage(),
      ]);
      return [];
    }
  }

  /**
   * Calculates security risk level.
   */
  protected function calculateSecurityRiskLevel(array $advisories, array $code_issues) {
    $risk_score = 0;

    foreach ($advisories as $advisory) {
      switch (strtolower($advisory['risk'] ?? 'unknown')) {
        case 'critical':
          $risk_score += 10;
          break;
        case 'high':
          $risk_score += 7;
          break;
        case 'moderate':
          $risk_score += 4;
          break;
        case 'low':
          $risk_score += 1;
          break;
      }
    }

    $risk_score += count($code_issues) * 2;

    if ($risk_score >= 10) {
      return 'critical';
    }
    elseif ($risk_score >= 7) {
      return 'high';
    }
    elseif ($risk_score >= 4) {
      return 'medium';
    }
    else {
      return 'low';
    }
  }

  /**
   * Performs basic code analysis without AI.
   *
   * @param array $ast
   *   The AST of the file.
   * @param string $file
   *   The file path.
   *
   * @return array
   *   Basic analysis results.
   */
  protected function performBasicAnalysis($ast, $file) {
    $results = [
      'status' => 'success',
      'file' => $file,
      'analysis' => [
        'deprecated_functions' => [],
        'deprecated_hooks' => [],
        'deprecated_classes' => [],
        'upgrade_suggestions' => [],
      ],
    ];

    // Traverse AST to find deprecated code
    $traverser = new \PhpParser\NodeTraverser();
    $visitor = new class extends \PhpParser\NodeVisitorAbstract {
      public $deprecated = [];
      
      public function enterNode(\PhpParser\Node $node) {
        if ($node instanceof \PhpParser\Node\Stmt\Function_) {
          // Check for deprecated functions
          foreach ($node->getAttributes() as $attr) {
            if (isset($attr['comments'])) {
              foreach ($attr['comments'] as $comment) {
                if (strpos($comment->getText(), '@deprecated') !== FALSE) {
                  $this->deprecated['functions'][] = $node->name->toString();
                }
              }
            }
          }
        }
        elseif ($node instanceof \PhpParser\Node\Stmt\Class_) {
          // Check for deprecated classes
          foreach ($node->getAttributes() as $attr) {
            if (isset($attr['comments'])) {
              foreach ($attr['comments'] as $comment) {
                if (strpos($comment->getText(), '@deprecated') !== FALSE) {
                  $this->deprecated['classes'][] = $node->name->toString();
                }
              }
            }
          }
        }
      }
    };
    
    $traverser->addVisitor($visitor);
    $traverser->traverse($ast);

    // Add deprecated items to results
    if (!empty($visitor->deprecated['functions'])) {
      $results['analysis']['deprecated_functions'] = $visitor->deprecated['functions'];
    }
    if (!empty($visitor->deprecated['classes'])) {
      $results['analysis']['deprecated_classes'] = $visitor->deprecated['classes'];
    }

    // Add basic upgrade suggestions
    $results['analysis']['upgrade_suggestions'][] = [
      'type' => 'info',
      'message' => 'Basic analysis completed. For more detailed analysis, please upgrade to a Pro or Enterprise subscription.',
    ];

    return $results;
  }
}
