<?php

namespace Drupal\Tests\ai_upgrade_assistant\Functional\Performance;

use Drupal\Tests\ai_upgrade_assistant\TestBase;

/**
 * Tests performance aspects of the module.
 *
 * @group ai_upgrade_assistant
 */
class PerformanceTest extends TestBase {

  /**
   * Tests parallel processing capabilities.
   */
  public function testParallelProcessing() {
    // Create multiple test projects.
    $projects = [];
    for ($i = 1; $i <= 5; $i++) {
      $projects[] = $this->createMockProject("test_module_$i");
    }

    // Start timer.
    $start_time = microtime(true);

    // Trigger parallel analysis.
    $batch = [
      'operations' => [],
      'finished' => '\Drupal\ai_upgrade_assistant\Service\BatchAnalyzer::finishBatch',
      'title' => $this->t('Analyzing projects in parallel'),
      'progress_message' => $this->t('Completed @current out of @total projects.'),
      'error_message' => $this->t('Error occurred during analysis.'),
    ];

    foreach ($projects as $project) {
      $batch['operations'][] = [
        '\Drupal\ai_upgrade_assistant\Service\BatchAnalyzer::processBatch',
        [$project],
      ];
    }

    batch_set($batch);
    $this->drupalGet('admin/reports/upgrade-status');
    $this->assertSession()->statusCodeEquals(200);

    // End timer.
    $end_time = microtime(true);
    $total_time = $end_time - $start_time;

    // Assert that parallel processing is faster than sequential.
    $this->assertLessThan(
      count($projects) * 5, // Expected sequential time
      $total_time,
      'Parallel processing should be significantly faster than sequential'
    );
  }

  /**
   * Tests memory usage optimization.
   */
  public function testMemoryUsage() {
    // Record initial memory.
    $initial_memory = memory_get_usage();

    // Create a large test project.
    $project_path = $this->createMockProject('large_test_module');
    
    // Add many files to test memory handling.
    for ($i = 1; $i <= 100; $i++) {
      file_put_contents(
        "$project_path/test_file_$i.php",
        "<?php\n// Large file $i\n" . str_repeat('// padding\n', 1000)
      );
    }

    // Analyze the large project.
    $this->drupalGet("admin/reports/upgrade-status/analyze/$project_path");
    $this->assertSession()->statusCodeEquals(200);

    // Check memory usage.
    $peak_memory = memory_get_peak_usage();
    $memory_increase = $peak_memory - $initial_memory;

    // Assert reasonable memory usage (less than 100MB increase).
    $this->assertLessThan(
      100 * 1024 * 1024,
      $memory_increase,
      'Memory usage should be reasonable for large projects'
    );
  }

  /**
   * Tests caching effectiveness.
   */
  public function testCaching() {
    // Create test project.
    $project_path = $this->createMockProject('cache_test_module');

    // First run - should be uncached.
    $start_time = microtime(true);
    $this->drupalGet("admin/reports/upgrade-status/analyze/$project_path");
    $this->assertSession()->statusCodeEquals(200);
    $uncached_time = microtime(true) - $start_time;

    // Second run - should use cache.
    $start_time = microtime(true);
    $this->drupalGet("admin/reports/upgrade-status/analyze/$project_path");
    $this->assertSession()->statusCodeEquals(200);
    $cached_time = microtime(true) - $start_time;

    // Assert cached response is significantly faster.
    $this->assertLessThan(
      $uncached_time / 2,
      $cached_time,
      'Cached response should be at least twice as fast as uncached'
    );
  }

  /**
   * Tests resource monitoring.
   */
  public function testResourceMonitoring() {
    // Enable resource monitoring.
    $this->config('ai_upgrade_assistant.settings')
      ->set('enable_resource_monitoring', TRUE)
      ->save();

    // Create test project.
    $project_path = $this->createMockProject('resource_test_module');

    // Analyze with resource monitoring.
    $this->drupalGet("admin/reports/upgrade-status/analyze/$project_path");
    $this->assertSession()->statusCodeEquals(200);

    // Check for resource monitoring data.
    $monitoring_data = \Drupal::state()->get('ai_upgrade_assistant.resource_monitoring');
    
    $this->assertNotEmpty($monitoring_data);
    $this->assertArrayHasKey('memory_usage', $monitoring_data);
    $this->assertArrayHasKey('execution_time', $monitoring_data);
    $this->assertArrayHasKey('cpu_usage', $monitoring_data);
  }
}
