<?php

namespace Drupal\Tests\ai_upgrade_assistant\Unit\Service\NodeVisitor;

use Drupal\Tests\UnitTestCase;
use PhpParser\Node;
use PhpParser\NodeTraverser;
use PhpParser\ParserFactory;
use Drupal\ai_upgrade_assistant\Service\NodeVisitor\DeprecatedFunctionVisitor;
use Drupal\ai_upgrade_assistant\Service\NodeVisitor\DeprecatedConstantVisitor;
use Drupal\ai_upgrade_assistant\Service\NodeVisitor\DeprecatedTraitVisitor;
use Drupal\ai_upgrade_assistant\Service\NodeVisitor\HookVisitor;
use Drupal\ai_upgrade_assistant\Service\NodeVisitor\ClassUsageVisitor;

/**
 * Tests the deprecation visitors.
 *
 * @group ai_upgrade_assistant
 */
class DeprecationVisitorsTest extends UnitTestCase {

  /**
   * The PHP parser.
   *
   * @var \PhpParser\Parser
   */
  protected $parser;

  /**
   * The node traverser.
   *
   * @var \PhpParser\NodeTraverser
   */
  protected $traverser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->parser = (new ParserFactory)->create(ParserFactory::PREFER_PHP7);
    $this->traverser = new NodeTraverser();
  }

  /**
   * Tests deprecated function detection.
   */
  public function testDeprecatedFunctionDetection() {
    $code = <<<'EOT'
<?php
drupal_set_message('Hello');
format_date(time());
node_load(1);
EOT;

    $visitor = new DeprecatedFunctionVisitor();
    $this->traverser->addVisitor($visitor);
    
    $ast = $this->parser->parse($code);
    $this->traverser->traverse($ast);
    
    $findings = $visitor->getFindings();
    $this->assertCount(3, $findings);
    
    $this->assertEquals('drupal_set_message', $findings[0]['name']);
    $this->assertEquals('format_date', $findings[1]['name']);
    $this->assertEquals('node_load', $findings[2]['name']);
    
    $this->assertTrue($findings[0]['critical']);
    $this->assertEquals('\Drupal::messenger()->addMessage()', $findings[0]['replacement']);
  }

  /**
   * Tests deprecated constant detection.
   */
  public function testDeprecatedConstantDetection() {
    $code = <<<'EOT'
<?php
$lang = LANGUAGE_NONE;
$status = FILE_STATUS_PERMANENT;
$time = REQUEST_TIME;
EOT;

    $visitor = new DeprecatedConstantVisitor();
    $this->traverser->addVisitor($visitor);
    
    $ast = $this->parser->parse($code);
    $this->traverser->traverse($ast);
    
    $findings = $visitor->getFindings();
    $this->assertCount(3, $findings);
    
    $this->assertEquals('LANGUAGE_NONE', $findings[0]['name']);
    $this->assertEquals('FILE_STATUS_PERMANENT', $findings[1]['name']);
    $this->assertEquals('REQUEST_TIME', $findings[2]['name']);
    
    $this->assertTrue($findings[0]['critical']);
    $this->assertEquals('LanguageInterface::LANGCODE_NOT_SPECIFIED', $findings[0]['replacement']);
  }

  /**
   * Tests deprecated trait detection.
   */
  public function testDeprecatedTraitDetection() {
    $code = <<<'EOT'
<?php
use Drupal\Core\Entity\EntityManagerTrait;
use Drupal\Core\Form\FormBuilderTrait;

class MyClass {
  use EntityManagerTrait;
  use FormBuilderTrait;
}
EOT;

    $visitor = new DeprecatedTraitVisitor();
    $this->traverser->addVisitor($visitor);
    
    $ast = $this->parser->parse($code);
    $this->traverser->traverse($ast);
    
    $findings = $visitor->getFindings();
    $this->assertCount(2, $findings);
    
    $this->assertEquals('Drupal\Core\Entity\EntityManagerTrait', $findings[0]['trait']);
    $this->assertEquals('Drupal\Core\Form\FormBuilderTrait', $findings[1]['trait']);
    
    $this->assertTrue($findings[0]['critical']);
    $this->assertEquals('Drupal\Core\Entity\EntityTypeManagerInterface', $findings[0]['replacement']);
  }

  /**
   * Tests deprecated hook detection.
   */
  public function testDeprecatedHookDetection() {
    $code = <<<'EOT'
<?php
function mymodule_entity_load($entities) {
}

function mymodule_entity_presave($entity) {
}
EOT;

    $visitor = new HookVisitor('mymodule');
    $this->traverser->addVisitor($visitor);
    
    $ast = $this->parser->parse($code);
    $this->traverser->traverse($ast);
    
    $findings = $visitor->getFindings();
    $this->assertCount(2, $findings);
    
    $this->assertEquals('hook_entity_load', $findings[0]['hook']);
    $this->assertEquals('hook_entity_presave', $findings[1]['hook']);
    
    $this->assertTrue($findings[0]['deprecated']);
    $this->assertEquals('hook_ENTITY_TYPE_load', $findings[0]['replacement']);
  }

  /**
   * Tests deprecated class detection.
   */
  public function testDeprecatedClassDetection() {
    $code = <<<'EOT'
<?php
use Drupal\Core\Entity\EntityNG;
use Drupal\field\Plugin\Type\Widget\WidgetBase;

class MyEntity extends EntityNG {
}

class MyWidget extends WidgetBase {
}

$entity = new EntityNG();
EOT;

    $visitor = new ClassUsageVisitor();
    $this->traverser->addVisitor($visitor);
    
    $ast = $this->parser->parse($code);
    $this->traverser->traverse($ast);
    
    $findings = $visitor->getFindings();
    $this->assertCount(3, $findings);
    
    $this->assertEquals('Drupal\Core\Entity\EntityNG', $findings[0]['extends']);
    $this->assertEquals('Drupal\field\Plugin\Type\Widget\WidgetBase', $findings[1]['extends']);
    $this->assertEquals('Drupal\Core\Entity\EntityNG', $findings[2]['class']);
    
    $this->assertTrue($findings[0]['deprecated']);
    $this->assertEquals('Drupal\Core\Entity\ContentEntityBase', $findings[0]['replacement']);
  }

  /**
   * Tests Drupal 11 changes detection.
   */
  public function testDrupal11ChangesDetection() {
    $code = <<<'EOT'
<?php
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Config\Entity\ConfigEntityInterface;

class MyEntity implements EntityTypeInterface, ConfigEntityInterface {
}
EOT;

    $visitor = new ClassUsageVisitor();
    $this->traverser->addVisitor($visitor);
    
    $ast = $this->parser->parse($code);
    $this->traverser->traverse($ast);
    
    $findings = $visitor->getFindings();
    $this->assertNotEmpty($findings);
    
    $found_d11_changes = false;
    foreach ($findings as $finding) {
      if (!empty($finding['drupal11_changes'])) {
        $found_d11_changes = true;
        break;
      }
    }
    
    $this->assertTrue($found_d11_changes, 'Should detect Drupal 11 changes');
  }

}
