<?php

namespace Drupal\Tests\ai_upgrade_assistant\Unit\Service\PatchManager;

use Drupal\ai_upgrade_assistant\Service\PatchManager;
use Drupal\ai_upgrade_assistant\Service\PatchValidator;
use Drupal\Tests\UnitTestCase;
use Drupal\Core\File\FileSystemInterface;
use Symfony\Component\Process\Process;

/**
 * Tests the patch manager service.
 *
 * @group ai_upgrade_assistant
 * @coversDefaultClass \Drupal\ai_upgrade_assistant\Service\PatchManager
 */
class PatchManagerTest extends UnitTestCase {

  /**
   * The patch manager service under test.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\PatchManager
   */
  protected $patchManager;

  /**
   * The mock file system.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The mock patch validator.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\PatchValidator
   */
  protected $patchValidator;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->fileSystem = $this->createMock(FileSystemInterface::class);
    $this->patchValidator = $this->createMock(PatchValidator::class);

    $this->patchManager = new PatchManager(
      $this->fileSystem,
      $this->patchValidator
    );
  }

  /**
   * Tests patch generation.
   *
   * @covers ::generatePatch
   */
  public function testPatchGeneration() {
    $original_file = '/path/to/original.php';
    $modified_file = '/path/to/modified.php';
    $patch_content = 'Test patch content';

    $process = $this->createMock(Process::class);
    $process->expects($this->once())
      ->method('run')
      ->willReturn(0);
    $process->expects($this->once())
      ->method('getOutput')
      ->willReturn($patch_content);

    $this->fileSystem->expects($this->once())
      ->method('saveData')
      ->with($patch_content)
      ->willReturn('patch_file.patch');

    $result = $this->patchManager->generatePatch($original_file, $modified_file);
    
    $this->assertIsArray($result);
    $this->assertArrayHasKey('patch_file', $result);
    $this->assertArrayHasKey('content', $result);
  }

  /**
   * Tests patch application.
   *
   * @covers ::applyPatch
   */
  public function testPatchApplication() {
    $patch_file = '/path/to/patch.patch';
    $target_dir = '/path/to/module';

    $this->patchValidator->expects($this->once())
      ->method('validatePatch')
      ->with($patch_file, $target_dir)
      ->willReturn(['valid' => true]);

    $process = $this->createMock(Process::class);
    $process->expects($this->once())
      ->method('run')
      ->willReturn(0);

    $result = $this->patchManager->applyPatch($patch_file, $target_dir);
    
    $this->assertTrue($result['success']);
  }

  /**
   * Tests conflict detection.
   *
   * @covers ::detectConflicts
   */
  public function testConflictDetection() {
    $patch_file = '/path/to/patch.patch';
    $target_dir = '/path/to/module';

    $process = $this->createMock(Process::class);
    $process->expects($this->once())
      ->method('run')
      ->willReturn(1);
    $process->expects($this->once())
      ->method('getErrorOutput')
      ->willReturn('Conflict detected');

    $result = $this->patchManager->detectConflicts($patch_file, $target_dir);
    
    $this->assertTrue($result['has_conflicts']);
    $this->assertArrayHasKey('conflicts', $result);
  }

  /**
   * Tests rollback functionality.
   *
   * @covers ::rollbackPatch
   */
  public function testPatchRollback() {
    $patch_file = '/path/to/patch.patch';
    $target_dir = '/path/to/module';

    $process = $this->createMock(Process::class);
    $process->expects($this->once())
      ->method('run')
      ->willReturn(0);

    $result = $this->patchManager->rollbackPatch($patch_file, $target_dir);
    
    $this->assertTrue($result['success']);
  }

  /**
   * Tests patch validation integration.
   *
   * @covers ::validateAndApplyPatch
   */
  public function testPatchValidationAndApplication() {
    $patch_file = '/path/to/patch.patch';
    $target_dir = '/path/to/module';

    $this->patchValidator->expects($this->once())
      ->method('validatePatch')
      ->willReturn([
        'valid' => true,
        'syntax_valid' => true,
        'security_issues' => [],
      ]);

    $result = $this->patchManager->validateAndApplyPatch($patch_file, $target_dir);
    
    $this->assertArrayHasKey('validation', $result);
    $this->assertArrayHasKey('application', $result);
  }
}
