<?php

namespace Drupal\ai_upgrade_assistant\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\ai_upgrade_assistant\Service\AchievementService;
use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;

/**
 * Controller for character-related pages.
 */
class CharacterController extends ControllerBase {

  /**
   * The achievement service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\AchievementService
   */
  protected $achievementService;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The date formatter.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * Constructs a new CharacterController object.
   *
   * @param \Drupal\ai_upgrade_assistant\Service\AchievementService $achievement_service
   *   The achievement service.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter.
   */
  public function __construct(
    AchievementService $achievement_service,
    Connection $database,
    DateFormatterInterface $date_formatter
  ) {
    $this->achievementService = $achievement_service;
    $this->database = $database;
    $this->dateFormatter = $date_formatter;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('ai_upgrade_assistant.achievement'),
      $container->get('database'),
      $container->get('date.formatter')
    );
  }

  /**
   * Displays the character sheet for a user.
   *
   * @param \Drupal\Core\Session\AccountInterface $user
   *   The user account.
   *
   * @return array
   *   A render array representing the character sheet page.
   */
  public function viewCharacterSheet(AccountInterface $user) {
    // Load achievements
    $achievements = $this->database->select('ai_upgrade_achievements', 'a')
      ->fields('a')
      ->condition('uid', $user->id())
      ->execute()
      ->fetchAll();

    // Load stats
    $stats = $this->database->select('ai_upgrade_stats', 's')
      ->fields('s')
      ->condition('uid', $user->id())
      ->execute()
      ->fetchObject();

    // Load guild if user is in one
    $guild = NULL;
    if ($stats && $stats->guild_id) {
      $guild = $this->database->select('ai_upgrade_guilds', 'g')
        ->fields('g')
        ->condition('id', $stats->guild_id)
        ->execute()
        ->fetchObject();
    }

    // Get recent achievements (last 5)
    $recent_achievements = [];
    foreach ($achievements as $achievement) {
      if (count($recent_achievements) >= 5) {
        break;
      }

      // Convert achievement data into a display-friendly format
      $recent_achievements[] = [
        'name' => $this->getAchievementName($achievement),
        'icon' => $this->getAchievementIcon($achievement),
        'rarity' => $this->getAchievementRarity($achievement),
        'date' => $achievement->created,
      ];
    }

    // Calculate next level XP requirement
    $current_achievement = reset($achievements);
    $next_level_xp = $current_achievement ? 
      $this->achievementService->getNextLevelRequirement($current_achievement->level) : 100;

    // Add our CSS
    $build['#attached']['library'][] = 'ai_upgrade_assistant/character_sheet';

    // Build the render array
    $build['character_sheet'] = [
      '#theme' => 'ai_upgrade_character_sheet',
      '#user' => $user,
      '#achievements' => $current_achievement,
      '#stats' => $stats,
      '#guild' => $guild,
      '#recent_achievements' => $recent_achievements,
      '#next_level_xp' => $next_level_xp,
      '#cache' => [
        'tags' => [
          'user:' . $user->id(),
          'ai_upgrade_achievements',
        ],
        'contexts' => ['user'],
        'max-age' => 300, // Cache for 5 minutes
      ],
    ];

    return $build;
  }

  /**
   * Gets a display name for an achievement.
   *
   * @param object $achievement
   *   The achievement object.
   *
   * @return string
   *   The achievement display name.
   */
  protected function getAchievementName($achievement) {
    $names = [
      'upgrade' => [
        'Successful Module Upgrade',
        'Pattern Master',
        'Code Savior',
        'Legendary Upgrader',
      ],
      'pattern' => [
        'Pattern Contributor',
        'Knowledge Keeper',
        'Wisdom Sharer',
        'Pattern Sage',
      ],
      'community' => [
        'Community Helper',
        'Guild Supporter',
        'Community Champion',
        'Legendary Guide',
      ],
    ];

    $type = $achievement->achievement_type;
    $level_index = min(floor($achievement->level / 5), 3);
    
    return $names[$type][$level_index] ?? 'Unknown Achievement';
  }

  /**
   * Gets an icon for an achievement.
   *
   * @param object $achievement
   *   The achievement object.
   *
   * @return string
   *   The Font Awesome icon name.
   */
  protected function getAchievementIcon($achievement) {
    $icons = [
      'upgrade' => 'sync-alt',
      'pattern' => 'lightbulb',
      'community' => 'users',
    ];

    return $icons[$achievement->achievement_type] ?? 'star';
  }

  /**
   * Gets the rarity class for an achievement.
   *
   * @param object $achievement
   *   The achievement object.
   *
   * @return string
   *   The rarity class.
   */
  protected function getAchievementRarity($achievement) {
    if ($achievement->level >= 15) {
      return 'legendary';
    }
    elseif ($achievement->level >= 10) {
      return 'epic';
    }
    elseif ($achievement->level >= 5) {
      return 'rare';
    }
    elseif ($achievement->level >= 2) {
      return 'uncommon';
    }
    return 'common';
  }

}
