<?php

namespace Drupal\ai_upgrade_assistant\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Drupal\ai_upgrade_assistant\Service\UpgradePathGenerator;
use Drupal\Core\Extension\ModuleHandlerInterface;

/**
 * Controller for upgrade path visualization and interaction.
 */
class UpgradePathController extends ControllerBase {

  /**
   * The upgrade path generator.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\UpgradePathGenerator
   */
  protected $pathGenerator;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Constructs a new UpgradePathController.
   */
  public function __construct(
    UpgradePathGenerator $path_generator,
    ModuleHandlerInterface $module_handler
  ) {
    $this->pathGenerator = $path_generator;
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('ai_upgrade_assistant.upgrade_path_generator'),
      $container->get('module_handler')
    );
  }

  /**
   * Checks access for viewing upgrade paths.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   Run access checks for this account.
   * @param string $module
   *   The module name.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(AccountInterface $account, $module = NULL) {
    if (!$module || !$this->moduleHandler->moduleExists($module)) {
      return AccessResult::forbidden();
    }
    return AccessResult::allowedIfHasPermission($account, 'access upgrade analysis');
  }

  /**
   * Displays the upgrade path visualization page.
   *
   * @param string $module
   *   The module name.
   *
   * @return array
   *   A render array.
   */
  public function viewPath($module) {
    $path = $this->pathGenerator->generateUpgradePath(
      $module,
      $this->getModuleVersion($module),
      $this->getTargetVersion()
    );

    return [
      '#theme' => 'upgrade_path',
      '#module' => $module,
      '#path' => $path,
      '#attached' => [
        'library' => [
          'ai_upgrade_assistant/upgrade_path_visualization',
        ],
      ],
    ];
  }

  /**
   * Displays the interactive upgrade path visualization.
   *
   * @param string $module
   *   The module name.
   *
   * @return array
   *   A render array.
   */
  public function viewInteractive($module) {
    $path = $this->pathGenerator->generateUpgradePath(
      $module,
      $this->getModuleVersion($module),
      $this->getTargetVersion()
    );

    return [
      '#theme' => 'upgrade_path_interactive',
      '#module' => $module,
      '#path' => $path,
      '#attached' => [
        'library' => [
          'ai_upgrade_assistant/upgrade_path_interactive',
          'ai_upgrade_assistant/d3',
        ],
        'drupalSettings' => [
          'aiUpgradeAssistant' => [
            'upgradePath' => $path,
          ],
        ],
      ],
    ];
  }

  /**
   * Exports the upgrade path.
   *
   * @param string $module
   *   The module name.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function exportPath($module, Request $request) {
    $format = $request->query->get('format', 'json');
    $path = $this->pathGenerator->generateUpgradePath(
      $module,
      $this->getModuleVersion($module),
      $this->getTargetVersion()
    );

    switch ($format) {
      case 'json':
        return new JsonResponse($path);

      case 'html':
        return [
          '#theme' => 'upgrade_path_export',
          '#module' => $module,
          '#path' => $path,
          '#attached' => [
            'library' => ['ai_upgrade_assistant/upgrade_path_export'],
          ],
        ];

      default:
        return new JsonResponse(['error' => 'Unsupported format'], 400);
    }
  }

  /**
   * Displays detailed information about a specific upgrade step.
   *
   * @param string $module
   *   The module name.
   * @param string $step
   *   The step ID.
   *
   * @return array
   *   A render array.
   */
  public function viewStep($module, $step) {
    $path = $this->pathGenerator->generateUpgradePath(
      $module,
      $this->getModuleVersion($module),
      $this->getTargetVersion()
    );

    // Find the specific step
    $step_data = NULL;
    foreach ($path['steps'] as $s) {
      if ($this->getStepId($s) === $step) {
        $step_data = $s;
        break;
      }
    }

    if (!$step_data) {
      throw new NotFoundHttpException('Step not found');
    }

    return [
      '#theme' => 'upgrade_path_step',
      '#module' => $module,
      '#step' => $step_data,
      '#attached' => [
        'library' => ['ai_upgrade_assistant/upgrade_path_step'],
      ],
    ];
  }

  /**
   * Gets the current version of a module.
   *
   * @param string $module
   *   The module name.
   *
   * @return string
   *   The module version.
   */
  protected function getModuleVersion($module) {
    $info = $this->moduleHandler->getModule($module)->info;
    return $info['version'] ?? '8.x-1.0';
  }

  /**
   * Gets the target Drupal version.
   *
   * @return string
   *   The target version.
   */
  protected function getTargetVersion() {
    // This could be made configurable
    return '9.0.0';
  }

  /**
   * Gets a unique identifier for a step.
   *
   * @param array $step
   *   The step data.
   *
   * @return string
   *   The step ID.
   */
  protected function getStepId($step) {
    return md5(serialize($step));
  }

}
