<?php

namespace Drupal\ai_upgrade_assistant\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Queue\QueueFactory;
use GuzzleHttp\ClientInterface;
use Drupal\Component\Serialization\Json;
use Drupal\ai_upgrade_assistant\Service\PatchValidator;
use Drupal\ai_upgrade_assistant\Service\AchievementService;

/**
 * Service for community learning and pattern sharing.
 */
class CommunityLearningService {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The queue factory.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected $queueFactory;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The pattern validator.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\PatchValidator|null
   */
  protected $patchValidator;

  /**
   * The achievement service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\AchievementService|null
   */
  protected $achievementService;

  /**
   * The forum topics table name.
   */
  const FORUM_TOPICS_TABLE = 'ai_upgrade_assistant_forum_topics';

  /**
   * The forum replies table name.
   */
  const FORUM_REPLIES_TABLE = 'ai_upgrade_assistant_forum_replies';

  /**
   * The pattern sharing table name.
   */
  const PATTERN_SHARING_TABLE = 'ai_upgrade_assistant_patterns';

  /**
   * The success stories table name.
   */
  const SUCCESS_STORIES_TABLE = 'ai_upgrade_assistant_success_stories';

  /**
   * Constructs a new CommunityLearningService.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    Connection $database,
    EntityTypeManagerInterface $entity_type_manager,
    AccountProxyInterface $current_user,
    StateInterface $state,
    LoggerChannelFactoryInterface $logger_factory,
    CacheBackendInterface $cache,
    QueueFactory $queue_factory,
    ClientInterface $http_client,
    ?PatchValidator $patch_validator = NULL,
    ?AchievementService $achievement_service = NULL
  ) {
    $this->configFactory = $config_factory;
    $this->database = $database;
    $this->entityTypeManager = $entity_type_manager;
    $this->currentUser = $current_user;
    $this->state = $state;
    $this->loggerFactory = $logger_factory;
    $this->cache = $cache;
    $this->queueFactory = $queue_factory;
    $this->httpClient = $http_client;
    $this->patchValidator = $patch_validator;
    $this->achievementService = $achievement_service;
  }

  /**
   * Creates a forum topic.
   *
   * @param string $title
   *   The topic title.
   * @param string $content
   *   The topic content.
   * @param array $tags
   *   Array of tags for the topic.
   * @param array $attachments
   *   Array of file attachments.
   *
   * @return int
   *   The topic ID.
   */
  public function createForumTopic($title, $content, array $tags = [], array $attachments = []) {
    $fields = [
      'title' => $title,
      'content' => $content,
      'uid' => $this->currentUser->id(),
      'created' => time(),
      'tags' => Json::encode($tags),
      'attachments' => Json::encode($attachments),
    ];

    $this->database->insert(self::FORUM_TOPICS_TABLE)
      ->fields($fields)
      ->execute();

    $topic_id = $this->database->lastInsertId();

    // Grant achievement for first forum post
    if ($this->isFirstForumPost($this->currentUser->id())) {
      if ($this->achievementService) {
        $this->achievementService->grantAchievement('first_forum_post');
      }
    }

    return $topic_id;
  }

  /**
   * Adds a reply to a forum topic.
   *
   * @param int $topic_id
   *   The topic ID.
   * @param string $content
   *   The reply content.
   * @param array $attachments
   *   Array of file attachments.
   *
   * @return int
   *   The reply ID.
   */
  public function addForumReply($topic_id, $content, array $attachments = []) {
    $fields = [
      'topic_id' => $topic_id,
      'content' => $content,
      'uid' => $this->currentUser->id(),
      'created' => time(),
      'attachments' => Json::encode($attachments),
    ];

    $this->database->insert(self::FORUM_REPLIES_TABLE)
      ->fields($fields)
      ->execute();

    return $this->database->lastInsertId();
  }

  /**
   * Shares a pattern with the community.
   *
   * @param string $title
   *   The pattern title.
   * @param string $description
   *   The pattern description.
   * @param array $pattern_data
   *   The pattern data.
   * @param array $tags
   *   Array of tags for the pattern.
   *
   * @return int
   *   The pattern ID.
   */
  public function sharePattern($title, $description, array $pattern_data, array $tags = []) {
    // Validate pattern before sharing
    if ($this->patchValidator) {
      $validation = $this->patchValidator->validatePattern($pattern_data);
      if (!$validation['valid']) {
        throw new \InvalidArgumentException('Invalid pattern: ' . implode(', ', $validation['errors']));
      }
    }

    $fields = [
      'title' => $title,
      'description' => $description,
      'pattern_data' => Json::encode($pattern_data),
      'uid' => $this->currentUser->id(),
      'created' => time(),
      'tags' => Json::encode($tags),
      'status' => 'pending_review',
    ];

    $this->database->insert(self::PATTERN_SHARING_TABLE)
      ->fields($fields)
      ->execute();

    $pattern_id = $this->database->lastInsertId();

    // Grant achievement for first pattern share
    if ($this->isFirstPatternShare($this->currentUser->id())) {
      if ($this->achievementService) {
        $this->achievementService->grantAchievement('first_pattern_share');
      }
    }

    return $pattern_id;
  }

  /**
   * Shares a success story.
   *
   * @param string $title
   *   The story title.
   * @param string $content
   *   The story content.
   * @param array $metadata
   *   Additional metadata about the success story.
   * @param array $screenshots
   *   Array of screenshot file IDs.
   *
   * @return int
   *   The story ID.
   */
  public function shareSuccessStory($title, $content, array $metadata = [], array $screenshots = []) {
    $fields = [
      'title' => $title,
      'content' => $content,
      'metadata' => Json::encode($metadata),
      'screenshots' => Json::encode($screenshots),
      'uid' => $this->currentUser->id(),
      'created' => time(),
    ];

    $this->database->insert(self::SUCCESS_STORIES_TABLE)
      ->fields($fields)
      ->execute();

    $story_id = $this->database->lastInsertId();

    // Grant achievement for first success story
    if ($this->isFirstSuccessStory($this->currentUser->id())) {
      if ($this->achievementService) {
        $this->achievementService->grantAchievement('first_success_story');
      }
    }

    return $story_id;
  }

  /**
   * Gets forum topics with optional filtering.
   *
   * @param array $filters
   *   Array of filters to apply.
   * @param int $limit
   *   Number of topics to return.
   * @param int $offset
   *   Offset for pagination.
   *
   * @return array
   *   Array of forum topics.
   */
  public function getForumTopics(array $filters = [], $limit = 10, $offset = 0) {
    $query = $this->database->select(self::FORUM_TOPICS_TABLE, 't')
      ->fields('t')
      ->orderBy('created', 'DESC')
      ->range($offset, $limit);

    foreach ($filters as $field => $value) {
      $query->condition($field, $value);
    }

    return $query->execute()->fetchAll();
  }

  /**
   * Get shared patterns for a specific context.
   *
   * @param array $config
   *   Configuration array containing:
   *   - task_type: Type of task (e.g., 'code_analysis')
   *   - context: Context for the patterns
   *
   * @return array
   *   Array of matching patterns
   */
  public function getSharedPatterns(array $config) {
    try {
      $query = $this->database->select(self::PATTERN_SHARING_TABLE, 'p')
        ->fields('p');

      // Check if task_type column exists before adding condition
      $schema = $this->database->schema();
      if ($schema->fieldExists(self::PATTERN_SHARING_TABLE, 'task_type')) {
        $query->condition('task_type', $config['task_type']);
      }
      
      $query->condition('context', $config['context'])
        ->orderBy('created', 'DESC')
        ->range(0, 10);

      return $query->execute()->fetchAll();
    }
    catch (\Exception $e) {
      // Log the error but return empty array to prevent breaking the application
      $this->loggerFactory->get('ai_upgrade_assistant')->error('Error fetching shared patterns: @error', ['@error' => $e->getMessage()]);
      return [];
    }
  }

  /**
   * Gets success stories with optional filtering.
   *
   * @param array $filters
   *   Array of filters to apply.
   * @param int $limit
   *   Number of stories to return.
   * @param int $offset
   *   Offset for pagination.
   *
   * @return array
   *   Array of success stories.
   */
  public function getSuccessStories(array $filters = [], $limit = 10, $offset = 0) {
    $query = $this->database->select(self::SUCCESS_STORIES_TABLE, 's')
      ->fields('s')
      ->orderBy('created', 'DESC')
      ->range($offset, $limit);

    foreach ($filters as $field => $value) {
      $query->condition($field, $value);
    }

    return $query->execute()->fetchAll();
  }

  /**
   * Checks if this is the user's first forum post.
   *
   * @param int $uid
   *   The user ID.
   *
   * @return bool
   *   TRUE if this is the first forum post.
   */
  protected function isFirstForumPost($uid) {
    return !$this->database->select(self::FORUM_TOPICS_TABLE, 't')
      ->condition('uid', $uid)
      ->countQuery()
      ->execute()
      ->fetchField();
  }

  /**
   * Checks if this is the user's first pattern share.
   *
   * @param int $uid
   *   The user ID.
   *
   * @return bool
   *   TRUE if this is the first pattern share.
   */
  protected function isFirstPatternShare($uid) {
    return !$this->database->select(self::PATTERN_SHARING_TABLE, 'p')
      ->condition('uid', $uid)
      ->countQuery()
      ->execute()
      ->fetchField();
  }

  /**
   * Checks if this is the user's first success story.
   *
   * @param int $uid
   *   The user ID.
   *
   * @return bool
   *   TRUE if this is the first success story.
   */
  protected function isFirstSuccessStory($uid) {
    return !$this->database->select(self::SUCCESS_STORIES_TABLE, 's')
      ->condition('uid', $uid)
      ->countQuery()
      ->execute()
      ->fetchField();
  }

  /**
   * Validates and approves a shared pattern.
   *
   * @param int $pattern_id
   *   The pattern ID.
   * @param string $reviewer_notes
   *   Notes from the reviewer.
   *
   * @return bool
   *   TRUE if pattern was approved.
   */
  public function approvePattern($pattern_id, $reviewer_notes = '') {
    $pattern = $this->database->select(self::PATTERN_SHARING_TABLE, 'p')
      ->fields('p')
      ->condition('id', $pattern_id)
      ->execute()
      ->fetchAssoc();

    if (!$pattern) {
      return FALSE;
    }

    // Re-validate pattern
    if ($this->patchValidator) {
      $pattern_data = Json::decode($pattern['pattern_data']);
      $validation = $this->patchValidator->validatePattern($pattern_data);
      if (!$validation['valid']) {
        return FALSE;
      }
    }

    $this->database->update(self::PATTERN_SHARING_TABLE)
      ->fields([
        'status' => 'approved',
        'reviewer_uid' => $this->currentUser->id(),
        'reviewer_notes' => $reviewer_notes,
        'approved' => time(),
      ])
      ->condition('id', $pattern_id)
      ->execute();

    // Grant achievement for pattern approval
    if ($this->achievementService) {
      $this->achievementService->grantAchievement('pattern_approved', $pattern['uid']);
    }

    return TRUE;
  }

  /**
   * Gets community statistics.
   *
   * @return array
   *   Array of community statistics.
   */
  public function getCommunityStats() {
    $stats = [];

    // Forum stats
    $stats['total_topics'] = $this->database->select(self::FORUM_TOPICS_TABLE)
      ->countQuery()
      ->execute()
      ->fetchField();

    $stats['total_replies'] = $this->database->select(self::FORUM_REPLIES_TABLE)
      ->countQuery()
      ->execute()
      ->fetchField();

    // Pattern stats
    $stats['total_patterns'] = $this->database->select(self::PATTERN_SHARING_TABLE)
      ->countQuery()
      ->execute()
      ->fetchField();

    $stats['approved_patterns'] = $this->database->select(self::PATTERN_SHARING_TABLE)
      ->condition('status', 'approved')
      ->countQuery()
      ->execute()
      ->fetchField();

    // Success story stats
    $stats['total_stories'] = $this->database->select(self::SUCCESS_STORIES_TABLE)
      ->countQuery()
      ->execute()
      ->fetchField();

    return $stats;
  }

  /**
   * Gets security advisories for a module.
   *
   * @param string $module_name
   *   The module name.
   *
   * @return array
   *   Array of security advisories with:
   *   - advisory_id: The security advisory ID
   *   - title: Title of the advisory
   *   - risk: Risk level
   *   - created: Creation timestamp
   *   - description: Description of the vulnerability
   *   - solution: Solution or mitigation steps
   *   - versions: Affected versions
   */
  public function getSecurityAdvisories($module_name) {
    $cid = 'ai_upgrade_assistant:security:' . $module_name;
    if ($cached = $this->cache->get($cid)) {
      return $cached->data;
    }

    $advisories = [];
    try {
      // First try the security API endpoint
      $response = $this->httpClient->get("https://www.drupal.org/api-d7/node.json", [
        'query' => [
          'type' => 'sa',
          'field_project' => $module_name,
          'sort' => 'created',
          'direction' => 'DESC',
        ],
      ]);

      $data = Json::decode((string) $response->getBody());
      
      if (!empty($data['list'])) {
        foreach ($data['list'] as $advisory) {
          $advisories[] = [
            'advisory_id' => $advisory['field_sa_identifier'] ?? '',
            'title' => $advisory['title'] ?? '',
            'risk' => $advisory['field_risk_level'] ?? 'Unknown',
            'created' => $advisory['created'] ?? 0,
            'description' => $advisory['field_sa_description'] ?? '',
            'solution' => $advisory['field_sa_solution'] ?? '',
            'versions' => $advisory['field_sa_affected_versions'] ?? [],
          ];
        }
      }

      // Also check security issues in the issue queue
      $issues_response = $this->httpClient->get("https://www.drupal.org/api-d7/node.json", [
        'query' => [
          'type' => 'project_issue',
          'field_project' => $module_name,
          'field_security_advisory' => 1,
          'sort' => 'changed',
          'direction' => 'DESC',
        ],
      ]);

      $issues_data = Json::decode((string) $issues_response->getBody());
      
      if (!empty($issues_data['list'])) {
        foreach ($issues_data['list'] as $issue) {
          $advisories[] = [
            'advisory_id' => 'ISSUE-' . ($issue['nid'] ?? ''),
            'title' => $issue['title'] ?? '',
            'risk' => 'Unknown',
            'created' => $issue['created'] ?? 0,
            'description' => $issue['field_issue_description'] ?? '',
            'solution' => 'See issue for details',
            'versions' => [],
            'issue_link' => 'https://www.drupal.org/node/' . ($issue['nid'] ?? ''),
          ];
        }
      }

      // Cache for 1 hour
      $this->cache->set($cid, $advisories, time() + 3600);

      return $advisories;
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('ai_upgrade_assistant')->error(
        'Error fetching security advisories for @module: @error',
        [
          '@module' => $module_name,
          '@error' => $e->getMessage(),
        ]
      );
      return [];
    }
  }

}
