<?php

namespace Drupal\ai_upgrade_assistant\Service;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;

/**
 * Service for analyzing Drupal projects for upgrades using AI.
 */
class ProjectAnalyzer {
  use DependencySerializationTrait;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The OpenAI service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\OpenAIService
   */
  protected $openAi;

  /**
   * The PHP parser service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\PhpParserService
   */
  protected $phpParser;

  /**
   * The AI model manager.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\AiModelManager
   */
  protected $aiModelManager;

  /**
   * The community learning service.
   *
   * @var \Drupal\ai_upgrade_assistant\Service\CommunityLearningService
   */
  protected $communityLearning;

  /**
   * Constructs a new ProjectAnalyzer.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   The file system service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory service.
   * @param \Drupal\ai_upgrade_assistant\Service\OpenAIService $openAi
   *   The OpenAI service.
   * @param \Drupal\ai_upgrade_assistant\Service\PhpParserService $phpParser
   *   The PHP parser service.
   * @param \Drupal\ai_upgrade_assistant\Service\AiModelManager $aiModelManager
   *   The AI model manager service.
   * @param \Drupal\ai_upgrade_assistant\Service\CommunityLearningService $communityLearning
   *   The community learning service.
   */
  public function __construct(
    ModuleHandlerInterface $moduleHandler,
    ConfigFactoryInterface $configFactory,
    FileSystemInterface $fileSystem,
    LoggerChannelFactoryInterface $loggerFactory,
    OpenAIService $openAi,
    PhpParserService $phpParser,
    AiModelManager $aiModelManager,
    CommunityLearningService $communityLearning
  ) {
    $this->moduleHandler = $moduleHandler;
    $this->configFactory = $configFactory;
    $this->fileSystem = $fileSystem;
    $this->logger = $loggerFactory->get('ai_upgrade_assistant');
    $this->openAi = $openAi;
    $this->phpParser = $phpParser;
    $this->aiModelManager = $aiModelManager;
    $this->communityLearning = $communityLearning;
  }

  /**
   * Analyzes a Drupal project for upgrade requirements.
   *
   * @param string $project_path
   *   Path to the project to analyze.
   *
   * @return array
   *   Analysis results.
   */
  public function analyzeProject($project_path) {
    // Get optimal model configuration for code analysis
    $model_config = $this->aiModelManager->getModelConfig('code_analysis', [
      'project_path' => $project_path,
    ]);

    // Find similar patterns from community
    $patterns = $this->communityLearning->getSharedPatterns([
      'project_path' => $project_path,
    ]);

    // Parse code and build AST
    $files = $this->findPhpFiles($project_path);
    $analysis_results = [];

    foreach ($files as $file) {
      $ast = $this->phpParser->parseFile($file);
      if (!$ast) {
        continue;
      }

      // Analyze code with AI
      $analysis = $this->analyzeCodeAst($ast, $file, $model_config, $patterns);
      if ($analysis) {
        $analysis_results[$file] = $analysis;
      }
    }

    // Record successful patterns
    foreach ($analysis_results as $file => $result) {
      if (!empty($result['patterns'])) {
        foreach ($result['patterns'] as $pattern) {
          $this->communityLearning->recordPattern('code_analysis', $pattern, [
            'file' => $file,
          ]);
        }
      }
    }

    return $analysis_results;
  }

  /**
   * Analyzes code using AI and pattern matching.
   *
   * @param array $ast
   *   The AST of the code.
   * @param string $file
   *   The file being analyzed.
   * @param array $model_config
   *   AI model configuration.
   * @param array $patterns
   *   Similar patterns from community.
   *
   * @return array|null
   *   Analysis results or null if no issues found.
   */
  protected function analyzeCodeAst(array $ast, $file, array $model_config, array $patterns) {
    // Run static analysis first
    $static_analysis = $this->analyzeWithVisitors($ast, $file);
    
    $code_context = $this->buildCodeContext($ast, $file);
    
    // Add static analysis results to context
    $code_context['static_analysis'] = $static_analysis;
    
    // Enhance context with community patterns
    if (!empty($patterns)) {
      $code_context['similar_patterns'] = array_column($patterns, 'pattern');
    }

    // Get AI analysis
    $analysis = $this->openAi->analyzeCode($code_context, $model_config);
    
    // Merge static and AI analysis
    $analysis['static_analysis'] = $static_analysis;
    
    // Record model performance
    $this->aiModelManager->recordModelPerformance(
      'code_analysis',
      $model_config,
      !empty($analysis['suggestions']),
      [
        'response_time' => $analysis['metrics']['duration'] ?? 0,
        'token_count' => $analysis['metrics']['tokens'] ?? 0,
        'static_findings' => count($static_analysis['critical_issues']),
      ]
    );

    return $analysis;
  }

  /**
   * Analyzes code for potential upgrade issues.
   *
   * @param string $path
   *   The path to analyze.
   *
   * @return array
   *   Analysis results.
   */
  public function analyzeCode($path) {
    try {
      $results = [];
      $files = $this->findPhpFiles($path);
      $model_config = $this->aiModelManager->getModelConfig('code_analysis', ['task_type' => 'code_analysis']);
      
      try {
        $patterns = $this->communityLearning->getSharedPatterns([
          'task_type' => 'code_analysis',
          'context' => 'code_analysis',
        ]);
      }
      catch (\Exception $e) {
        $this->logger->warning('Failed to fetch patterns: @error', ['@error' => $e->getMessage()]);
        $patterns = [];
      }

      foreach ($files as $file) {
        try {
          $code = file_get_contents($file);
          if ($code === FALSE) {
            $this->logger->warning('Could not read file: @file', ['@file' => $file]);
            continue;
          }

          $file_config = $model_config;
          if (!empty($patterns)) {
            $file_config['patterns'] = array_map(function ($pattern) {
              return is_object($pattern) && isset($pattern->pattern_data) 
                ? unserialize($pattern->pattern_data) 
                : [];
            }, $patterns);
          }

          $results[$file] = $this->openAi->analyzeCode($code, $file_config);
        }
        catch (\Exception $e) {
          $this->logger->error('Error analyzing file @file: @error', [
            '@file' => $file,
            '@error' => $e->getMessage(),
          ]);
          $results[$file] = [
            'status' => 'error',
            'message' => $e->getMessage(),
          ];
        }
      }

      return $results;
    }
    catch (\Exception $e) {
      $this->logger->error('Error in code analysis: @error', ['@error' => $e->getMessage()]);
      return [
        'status' => 'error',
        'message' => $e->getMessage(),
      ];
    }
  }

  /**
   * Analyzes dependencies of a module.
   *
   * @param string $module
   *   The module name.
   *
   * @return array
   *   Analysis of dependencies.
   */
  public function analyzeDependencies($module) {
    $module_data = $this->moduleHandler->getModule($module);
    if (!$module_data) {
      return [
        'status' => 'error',
        'message' => $this->t('Module @module not found', ['@module' => $module]),
      ];
    }

    $info = $module_data->getType() == 'module' ? $module_data->info : [];
    $dependencies = $info['dependencies'] ?? [];
    $results = [];

    foreach ($dependencies as $dependency) {
      // Parse dependency string (e.g., "drupal:views (>8.x-3.0)")
      if (preg_match('/^([a-z0-9_]+):([a-z0-9_]+)(?:\s*\(([^)]+)\))?$/', $dependency, $matches)) {
        $provider = $matches[1];
        $dep_name = $matches[2];
        $version_constraint = $matches[3] ?? '';

        $dep_module = $this->moduleHandler->getModule($dep_name);
        $status = 'unknown';
        $message = '';

        if ($dep_module) {
          $dep_info = $dep_module->getType() == 'module' ? $dep_module->info : [];
          $current_version = $dep_info['version'] ?? '';

          if ($current_version) {
            if ($this->isModuleCompatible($dep_name)) {
              $status = 'compatible';
              $message = $this->t('Compatible with Drupal 11');
            }
            else {
              $status = 'needs_update';
              $message = $this->t('Needs update for Drupal 11 compatibility');
            }
          }
        }
        else {
          $status = 'missing';
          $message = $this->t('Dependency not found');
        }

        $results[$dependency] = [
          'name' => $dep_name,
          'provider' => $provider,
          'version_constraint' => $version_constraint,
          'status' => $status,
          'message' => $message,
        ];
      }
    }

    return [
      'dependencies' => $results,
      'count' => count($results),
      'status' => count($results) > 0 ? 'has_dependencies' : 'no_dependencies',
    ];
  }

  /**
   * Analyzes security issues for a module.
   *
   * @param string $module
   *   The module name.
   *
   * @return array
   *   Security analysis results.
   */
  public function analyzeSecurityIssues($module) {
    try {
      $module_data = $this->moduleHandler->getModule($module);
      if (!$module_data) {
        return [
          'status' => 'error',
          'message' => $this->t('Module @module not found', ['@module' => $module]),
        ];
      }

      // Get known security advisories
      $advisories = $this->communityLearning->getSecurityAdvisories($module);
      
      // Analyze code for potential security issues
      $path = $module_data->getPath();
      $files = $this->findPhpFiles($path);
      $code_issues = [];
      
      foreach ($files as $file) {
        $code = file_get_contents($file);
        if ($code === FALSE) {
          continue;
        }
        
        $ast = $this->phpParser->parseCode($code);
        $file_issues = $this->analyzeCodeSecurity($ast, $file);
        
        if (!empty($file_issues)) {
          $code_issues[$file] = $file_issues;
        }
      }

      return [
        'status' => 'analyzed',
        'advisories' => [
          'count' => count($advisories),
          'items' => $advisories,
        ],
        'code_issues' => [
          'count' => count($code_issues),
          'items' => $code_issues,
        ],
        'risk_level' => $this->calculateSecurityRiskLevel($advisories, $code_issues),
      ];
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('ai_upgrade_assistant')->error(
        'Error analyzing security for @module: @error',
        [
          '@module' => $module,
          '@error' => $e->getMessage(),
        ]
      );
      return [
        'status' => 'error',
        'message' => $this->t('Error analyzing security: @error', ['@error' => $e->getMessage()]),
      ];
    }
  }

  /**
   * Analyzes code for security issues.
   *
   * @param array $ast
   *   The AST of the code.
   * @param string $file
   *   The file being analyzed.
   *
   * @return array
   *   Array of security issues found.
   */
  protected function analyzeCodeSecurity(array $ast, $file) {
    $issues = [];

    // Common security patterns to check
    $patterns = [
      'sql_injection' => [
        'functions' => ['db_query', 'db_query_range'],
        'message' => 'Potential SQL injection vulnerability. Use parameterized queries.',
      ],
      'xss' => [
        'functions' => ['t', 'drupal_set_message'],
        'message' => 'Potential XSS vulnerability. Use proper escaping.',
      ],
      'csrf' => [
        'functions' => ['drupal_get_form'],
        'message' => 'Form submission should include CSRF protection.',
      ],
      'file_access' => [
        'functions' => ['file_get_contents', 'fopen', 'file_put_contents'],
        'message' => 'Ensure proper file access checks are in place.',
      ],
    ];

    foreach ($patterns as $type => $pattern) {
      $found = $this->findFunctionCalls($ast, $pattern['functions']);
      if (!empty($found)) {
        foreach ($found as $function => $locations) {
          $issues[] = [
            'type' => $type,
            'severity' => 'warning',
            'message' => $pattern['message'],
            'function' => $function,
            'locations' => $locations,
          ];
        }
      }
    }

    return $issues;
  }

  /**
   * Calculates overall security risk level.
   *
   * @param array $advisories
   *   Security advisories.
   * @param array $code_issues
   *   Code security issues.
   *
   * @return string
   *   Risk level: low, medium, high, or critical.
   */
  protected function calculateSecurityRiskLevel(array $advisories, array $code_issues) {
    $risk_score = 0;

    // Score advisories
    foreach ($advisories as $advisory) {
      switch (strtolower($advisory['risk'] ?? 'unknown')) {
        case 'critical':
          $risk_score += 10;
          break;
        case 'high':
          $risk_score += 7;
          break;
        case 'moderate':
          $risk_score += 4;
          break;
        case 'low':
          $risk_score += 1;
          break;
      }
    }

    // Score code issues
    $risk_score += count($code_issues) * 2;

    // Determine risk level
    if ($risk_score >= 10) {
      return 'critical';
    }
    elseif ($risk_score >= 7) {
      return 'high';
    }
    elseif ($risk_score >= 4) {
      return 'medium';
    }
    else {
      return 'low';
    }
  }

  /**
   * Analyzes a single PHP file.
   *
   * @param string $file
   *   Path to the PHP file.
   *
   * @return array
   *   Analysis results for the file.
   */
  public function analyzeFile($file) {
    $code = file_get_contents($file);
    $ast = $this->phpParser->parseCode($code);
    
    return [
      'deprecated_functions' => $this->findDeprecatedFunctions($ast),
      'api_changes' => $this->findApiChanges($ast),
      'complexity_score' => $this->calculateComplexity($ast),
    ];
  }

  /**
   * Builds context for code analysis.
   *
   * @param array $ast
   *   The AST of the code.
   * @param string $file
   *   The file being analyzed.
   *
   * @return array
   *   Code context for analysis.
   */
  protected function buildCodeContext(array $ast, $file) {
    return [
      'file_path' => $file,
      'ast' => $ast,
      'drupal_version' => \Drupal::VERSION,
      'module_info' => $this->getModuleInfo($file),
      'dependencies' => $this->getDependencies($file),
    ];
  }

  /**
   * Gets module info for a file.
   *
   * @param string $file
   *   File path.
   *
   * @return array
   *   Module information.
   */
  protected function getModuleInfo($file) {
    $module_name = basename(dirname($file));
    return $this->moduleHandler->getModule($module_name)->info ?? [];
  }

  /**
   * Gets dependencies for a file.
   *
   * @param string $file
   *   File path.
   *
   * @return array
   *   Dependencies information.
   */
  protected function getDependencies($file) {
    $module_name = basename(dirname($file));
    $module = $this->moduleHandler->getModule($module_name);
    
    return [
      'dependencies' => $module->info['dependencies'] ?? [],
      'required_by' => $this->moduleHandler->getRequiredBy($module_name),
    ];
  }

  /**
   * Finds all PHP files in a project.
   *
   * @param string $project_path
   *   Path to the project.
   *
   * @return array
   *   List of PHP files.
   */
  protected function findPhpFiles($project_path) {
    $files = [];
    $iterator = new \RecursiveIteratorIterator(
      new \RecursiveDirectoryIterator($project_path)
    );

    foreach ($iterator as $file) {
      if ($file->isFile() && $file->getExtension() === 'php') {
        $files[] = $file->getPathname();
      }
    }

    return $files;
  }

  /**
   * Analyzes hook implementations in a module.
   *
   * @param string $module_name
   *   The name of the module to analyze.
   *
   * @return array
   *   Analysis results containing:
   *   - deprecated_hooks: List of deprecated hooks found
   *   - drupal11_attention: Hooks needing special attention in D11
   *   - update_hooks: Update hooks with potential issues
   *   - critical_issues: Critical hook implementation issues
   */
  protected function analyzeHookImplementations($module_name) {
    $module_path = $this->moduleHandler->getModule($module_name)->getPath();
    $files = $this->fileSystem->scanDirectory($module_path, '/\.module$|\.install$|\.inc$/');
    
    $results = [
      'deprecated_hooks' => [],
      'drupal11_attention' => [],
      'update_hooks' => [],
      'critical_issues' => [],
    ];

    foreach ($files as $file) {
      try {
        $visitor = new NodeVisitor\HookVisitor($module_name);
        $stmts = $this->phpParser->parseFile($file->uri);
        $this->phpParser->traverseNodes($stmts, [$visitor]);
        
        foreach ($visitor->getFindings() as $finding) {
          // Handle deprecated hooks
          if (!empty($finding['deprecated'])) {
            $issue = [
              'file' => $file->uri,
              'line' => $finding['line'],
              'hook' => $finding['hook'],
              'replacement' => $finding['replacement'],
              'version' => $finding['version'],
            ];
            
            if (!empty($finding['critical'])) {
              $results['critical_issues'][] = $issue;
            }
            
            $results['deprecated_hooks'][] = $issue;
          }

          // Handle hooks needing Drupal 11 attention
          if (!empty($finding['drupal11_attention'])) {
            $results['drupal11_attention'][] = [
              'file' => $file->uri,
              'line' => $finding['line'],
              'hook' => $finding['hook'],
              'note' => $finding['attention_note'],
              'example' => $finding['example'],
            ];
          }

          // Handle update hook issues
          if (!empty($finding['is_update_hook']) && !empty($finding['update_hook_warning'])) {
            $results['update_hooks'][] = [
              'file' => $file->uri,
              'line' => $finding['line'],
              'hook' => $finding['name'],
              'warning' => $finding['update_hook_warning'],
            ];
          }
        }
      }
      catch (\Exception $e) {
        $this->logger->error('Error analyzing hooks in @file: @message', [
          '@file' => $file->uri,
          '@message' => $e->getMessage(),
        ]);
      }
    }

    return $results;
  }

  /**
   * Analyzes code using static analysis visitors.
   *
   * @param array $ast
   *   The AST of the code.
   * @param string $file
   *   The file being analyzed.
   *
   * @return array
   *   Static analysis results containing:
   *   - deprecated_functions: List of deprecated function calls
   *   - deprecated_classes: List of deprecated class usage
   *   - deprecated_constants: List of deprecated constant usage
   *   - deprecated_traits: List of deprecated trait usage
   *   - deprecated_hooks: List of deprecated hook implementations
   *   - drupal11_attention: Items needing attention for D11
   *   - critical_issues: Critical deprecation issues
   *   - namespace_issues: Namespace and PSR-4 issues
   *   - service_issues: Service usage and dependency injection issues
   *   - event_issues: Event subscriber and deprecated events issues
   *   - plugin_issues: Plugin usage and deprecated plugin types
   */
  protected function analyzeWithVisitors(array $ast, $file) {
    $results = [
      'deprecated_functions' => [],
      'deprecated_classes' => [],
      'deprecated_constants' => [],
      'deprecated_traits' => [],
      'deprecated_hooks' => [],
      'drupal11_attention' => [],
      'critical_issues' => [],
      'namespace_issues' => [],
      'service_issues' => [],
      'event_issues' => [],
      'plugin_issues' => [],
    ];

    // Get module info
    $module_name = $this->getModuleNameFromPath($file);
    $module_path = dirname($file);

    // Initialize visitors
    $visitors = [
      new NodeVisitor\DeprecatedFunctionVisitor(),
      new NodeVisitor\ClassUsageVisitor(),
      new NodeVisitor\DeprecatedConstantVisitor(),
      new NodeVisitor\DeprecatedTraitVisitor(),
      new NodeVisitor\HookVisitor($module_name),
      new NodeVisitor\NamespaceVisitor($module_name, $module_path),
      new NodeVisitor\ServiceVisitor(),
      new NodeVisitor\EventSubscriberVisitor(),
      new NodeVisitor\PluginVisitor(),
    ];

    // Traverse AST with all visitors
    foreach ($visitors as $visitor) {
      $this->phpParser->traverseNodes($ast, [$visitor]);
      
      foreach ($visitor->getFindings() as $finding) {
        // Add file context to finding
        $finding['file'] = $file;
        
        // Sort findings by type
        switch ($finding['type']) {
          case 'deprecated_function':
            $results['deprecated_functions'][] = $finding;
            break;
            
          case 'class_extends':
          case 'implements_interface':
          case 'class_instantiation':
            if (!empty($finding['deprecated'])) {
              $results['deprecated_classes'][] = $finding;
            }
            break;
            
          case 'deprecated_constant':
            $results['deprecated_constants'][] = $finding;
            break;
            
          case 'trait_use':
            if (!empty($finding['deprecated'])) {
              $results['deprecated_traits'][] = $finding;
            }
            break;
            
          case 'hook':
            if (!empty($finding['deprecated'])) {
              $results['deprecated_hooks'][] = $finding;
            }
            break;

          case 'namespace':
            if (!empty($finding['psr4_violation']) || !empty($finding['issues']) || !empty($finding['deprecated_uses'])) {
              $results['namespace_issues'][] = $finding;
            }
            break;

          case 'static_service':
          case 'service_injection':
            if (!empty($finding['deprecated']) || !empty($finding['deprecated_services'])) {
              $results['service_issues'][] = $finding;
            }
            break;

          case 'event_subscriber':
            if (!empty($finding['deprecated_events'])) {
              $results['event_issues'][] = $finding;
            }
            break;

          case 'plugin':
            if (!empty($finding['deprecated'])) {
              $results['plugin_issues'][] = $finding;
            }
            break;
        }

        // Track critical issues
        if (!empty($finding['critical']) || 
            (!empty($finding['deprecated_services']) && 
             array_filter($finding['deprecated_services'], function($s) { return $s['critical']; })) ||
            (!empty($finding['deprecated_events']) && 
             array_filter($finding['deprecated_events'], function($e) { return $e['critical']; }))) {
          $results['critical_issues'][] = $finding;
        }

        // Track Drupal 11 attention items
        if (!empty($finding['drupal11_attention']) || 
            !empty($finding['drupal11_changes']) || 
            !empty($finding['drupal11_services']) ||
            !empty($finding['drupal11_events'])) {
          $results['drupal11_attention'][] = $finding;
        }
      }

      $visitor->reset();
    }

    return $results;
  }

  /**
   * Gets the module name from a file path.
   *
   * @param string $file_path
   *   The file path.
   *
   * @return string
   *   The module name.
   */
  protected function getModuleNameFromPath($file_path) {
    $parts = explode('/', $file_path);
    foreach ($parts as $part) {
      if (strpos($part, '.info.yml') !== false) {
        return str_replace('.info.yml', '', $part);
      }
      if (strpos($part, '.module') !== false) {
        return str_replace('.module', '', $part);
      }
    }
    return basename(dirname($file_path));
  }

}
