<?php

namespace Drupal\ai_upgrade_assistant\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Database\Connection;
use GuzzleHttp\ClientInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Manages Drupal Association integration and subscription features.
 *
 * This service integrates with Drupal.org's infrastructure to provide:
 * - Drupal Association member benefits
 * - Organization verification
 * - Usage tracking and quotas
 * - Premium feature access
 */
class SubscriptionManager {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Constructs a new SubscriptionManager.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    AccountInterface $current_user,
    StateInterface $state,
    Connection $database,
    ClientInterface $http_client,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->configFactory = $config_factory;
    $this->currentUser = $current_user;
    $this->state = $state;
    $this->database = $database;
    $this->httpClient = $http_client;
    $this->loggerFactory = $logger_factory->get('ai_upgrade_assistant');
  }

  /**
   * Gets the current subscription tier.
   *
   * Checks:
   * 1. Drupal Association membership status
   * 2. Organization verification
   * 3. Local site configuration
   *
   * @return string
   *   The subscription tier: 'community', 'supporter', 'organization'.
   */
  public function getCurrentTier(): string {
    // Check Drupal Association membership.
    if ($this->isDrupalAssociationMember()) {
      if ($this->isVerifiedOrganization()) {
        return 'organization';
      }
      return 'supporter';
    }
    return 'community';
  }

  /**
   * Checks if the current user/site has Drupal Association membership.
   *
   * @return bool
   *   TRUE if the user/site has an active DA membership.
   */
  public function isDrupalAssociationMember(): bool {
    try {
      // Check local cache first.
      $cid = 'ai_upgrade_assistant:da_membership:' . $this->currentUser->id();
      if ($cached = $this->state->get($cid)) {
        return $cached['status'] && $cached['expires'] > time();
      }

      // Query Drupal.org's API.
      $response = $this->httpClient->get('https://www.drupal.org/api-d7/user/' . $this->getDrupalOrgUid() . '/membership');
      $data = json_decode($response->getBody(), TRUE);

      // Cache the result.
      $this->state->set($cid, [
        'status' => !empty($data['membership_active']),
        'expires' => time() + 86400,
      ]);

      return !empty($data['membership_active']);
    }
    catch (\Exception $e) {
      $this->loggerFactory->error('Failed to check DA membership: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Checks if this is a verified Drupal organization.
   *
   * @return bool
   *   TRUE if this is a verified organization.
   */
  public function isVerifiedOrganization(): bool {
    try {
      // Check local cache first.
      $cid = 'ai_upgrade_assistant:org_verification';
      if ($cached = $this->state->get($cid)) {
        return $cached['verified'] && $cached['expires'] > time();
      }

      // Query Drupal.org's API.
      $response = $this->httpClient->get('https://www.drupal.org/api-d7/organization/' . $this->getDrupalOrgUid());
      $data = json_decode($response->getBody(), TRUE);

      // Cache the result.
      $this->state->set($cid, [
        'verified' => !empty($data['verified']),
        'expires' => time() + 86400,
      ]);

      return !empty($data['verified']);
    }
    catch (\Exception $e) {
      $this->loggerFactory->error('Failed to check organization status: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Gets the rate limit for the current tier.
   *
   * @return array
   *   Rate limit info with 'requests' and 'window' in seconds.
   */
  public function getRateLimit(): array {
    $tier = $this->getCurrentTier();
    switch ($tier) {
      case 'organization':
        return [
          'requests' => 1000,
          'window' => 3600,
        ];
      case 'supporter':
        return [
          'requests' => 500,
          'window' => 3600,
        ];
      default:
        return [
          'requests' => 100,
          'window' => 3600,
        ];
    }
  }

  /**
   * Gets the feature access for the current tier.
   *
   * @return array
   *   Array of feature flags.
   */
  public function getFeatureAccess(): array {
    $tier = $this->getCurrentTier();
    $features = [
      'batch_processing' => FALSE,
      'priority_support' => FALSE,
      'advanced_analytics' => FALSE,
      'custom_patterns' => FALSE,
    ];

    switch ($tier) {
      case 'organization':
        return array_merge($features, [
          'batch_processing' => TRUE,
          'priority_support' => TRUE,
          'advanced_analytics' => TRUE,
          'custom_patterns' => TRUE,
        ]);

      case 'supporter':
        return array_merge($features, [
          'batch_processing' => TRUE,
          'advanced_analytics' => TRUE,
        ]);

      default:
        return $features;
    }
  }

  /**
   * Gets the Drupal.org UID for the current user/site.
   *
   * @return int|null
   *   The Drupal.org UID or NULL if not found.
   */
  protected function getDrupalOrgUid(): ?int {
    return $this->configFactory
      ->get('ai_upgrade_assistant.settings')
      ->get('drupal_org_uid');
  }

}
