<?php

namespace Drupal\ai_upgrade_assistant\Theme;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Template\Attribute;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides preprocessing for AI Upgrade Assistant theme elements.
 */
class AiUpgradeAssistantPreprocess implements ContainerInjectionInterface {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new AiUpgradeAssistantPreprocess object.
   */
  public function __construct(ConfigFactoryInterface $configFactory) {
    $this->configFactory = $configFactory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory')
    );
  }

  /**
   * Preprocesses the upgrade dashboard.
   */
  public function preprocessDashboard(&$variables) {
    $config = $this->configFactory->get('ai_upgrade_assistant.settings');
    
    // Add theme settings
    $variables['dashboard_theme'] = $config->get('dashboard_theme') ?? 'modern';
    $variables['show_animations'] = $config->get('show_animations') ?? TRUE;
    $variables['compact_view'] = $config->get('compact_view') ?? FALSE;
    
    // Add dashboard attributes
    $variables['dashboard_attributes'] = new Attribute([
      'class' => [
        'ai-upgrade-dashboard',
        'theme-' . $variables['dashboard_theme'],
        $variables['show_animations'] ? 'with-animations' : 'no-animations',
        $variables['compact_view'] ? 'compact-view' : 'full-view',
      ],
    ]);
    
    // Add dynamic sections based on user permissions
    $variables['sections'] = $this->getDashboardSections();
    
    // Add custom CSS variables for theming
    $variables['#attached']['drupalSettings']['aiUpgradeAssistant']['theme'] = [
      'primaryColor' => $config->get('primary_color') ?? '#0078d4',
      'secondaryColor' => $config->get('secondary_color') ?? '#2ecc71',
      'fontFamily' => $config->get('font_family') ?? 'system-ui, -apple-system, sans-serif',
      'borderRadius' => $config->get('border_radius') ?? '4px',
    ];
  }

  /**
   * Preprocesses the upgrade analysis results.
   */
  public function preprocessAnalysisResults(&$variables) {
    $config = $this->configFactory->get('ai_upgrade_assistant.settings');
    
    // Add result display settings
    $variables['show_code_previews'] = $config->get('show_code_previews') ?? TRUE;
    $variables['group_by_severity'] = $config->get('group_by_severity') ?? TRUE;
    $variables['inline_diffs'] = $config->get('inline_diffs') ?? TRUE;
    
    // Add custom styling for different severity levels
    $variables['severity_colors'] = [
      'critical' => $config->get('critical_color') ?? '#e74c3c',
      'warning' => $config->get('warning_color') ?? '#f1c40f',
      'info' => $config->get('info_color') ?? '#3498db',
      'success' => $config->get('success_color') ?? '#2ecc71',
    ];
  }

  /**
   * Gets dashboard sections based on user permissions.
   *
   * @return array
   *   Array of dashboard sections with their settings.
   */
  protected function getDashboardSections() {
    $sections = [];
    
    // Module Overview Section
    $sections['overview'] = [
      'title' => t('Module Overview'),
      'weight' => 0,
      'collapsed' => FALSE,
    ];
    
    // Upgrade Analysis Section
    if (\Drupal::currentUser()->hasPermission('run upgrade analysis')) {
      $sections['analysis'] = [
        'title' => t('Upgrade Analysis'),
        'weight' => 10,
        'collapsed' => FALSE,
      ];
    }
    
    // Pattern Learning Section
    if (\Drupal::currentUser()->hasPermission('manage upgrade patterns')) {
      $sections['patterns'] = [
        'title' => t('Pattern Learning'),
        'weight' => 20,
        'collapsed' => TRUE,
      ];
    }
    
    // Community Insights Section
    if (\Drupal::currentUser()->hasPermission('view community insights')) {
      $sections['community'] = [
        'title' => t('Community Insights'),
        'weight' => 30,
        'collapsed' => TRUE,
      ];
    }
    
    // Sort sections by weight
    uasort($sections, function ($a, $b) {
      return $a['weight'] <=> $b['weight'];
    });
    
    return $sections;
  }

}
