<?php

namespace Drupal\Tests\ai_upgrade_assistant\Unit\Service\NodeVisitor;

use Drupal\Tests\UnitTestCase;
use PhpParser\NodeTraverser;
use PhpParser\ParserFactory;
use Drupal\ai_upgrade_assistant\Service\NodeVisitor\EventSubscriberVisitor;

/**
 * Tests the event subscriber visitor.
 *
 * @group ai_upgrade_assistant
 */
class EventSubscriberVisitorTest extends UnitTestCase {

  /**
   * The PHP parser.
   *
   * @var \PhpParser\Parser
   */
  protected $parser;

  /**
   * The node traverser.
   *
   * @var \PhpParser\NodeTraverser
   */
  protected $traverser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->parser = (new ParserFactory)->create(ParserFactory::PREFER_PHP7);
    $this->traverser = new NodeTraverser();
  }

  /**
   * Tests event subscriber detection.
   */
  public function testEventSubscriberDetection() {
    $code = <<<'EOT'
<?php
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class MyEventSubscriber implements EventSubscriberInterface {
  public static function getSubscribedEvents() {
    return [
      'kernel.request' => ['onKernelRequest', 100],
      'config.save' => 'onConfigSave',
      'kernel.view' => ['onView', 0],
    ];
  }
}
EOT;

    $visitor = new EventSubscriberVisitor();
    $this->traverser->addVisitor($visitor);
    
    $ast = $this->parser->parse($code);
    $this->traverser->traverse($ast);
    
    $findings = $visitor->getFindings();
    $this->assertCount(1, $findings);
    
    $this->assertEquals('event_subscriber', $findings[0]['type']);
    $this->assertEquals('MyEventSubscriber', $findings[0]['class']);
    $this->assertContains('kernel.request', $findings[0]['subscribed_events']);
  }

  /**
   * Tests deprecated event detection.
   */
  public function testDeprecatedEventDetection() {
    $code = <<<'EOT'
<?php
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class MyEventSubscriber implements EventSubscriberInterface {
  public static function getSubscribedEvents() {
    return [
      'config.save' => 'onConfigSave',
      'config.delete' => 'onConfigDelete',
    ];
  }
}
EOT;

    $visitor = new EventSubscriberVisitor();
    $this->traverser->addVisitor($visitor);
    
    $ast = $this->parser->parse($code);
    $this->traverser->traverse($ast);
    
    $findings = $visitor->getFindings();
    $this->assertNotEmpty($findings);
    
    $this->assertNotEmpty($findings[0]['deprecated_events']);
    $this->assertEquals('config.save', $findings[0]['deprecated_events'][0]['event']);
    $this->assertEquals('Drupal\Core\Config\ConfigEvents::SAVE', $findings[0]['deprecated_events'][0]['replacement']);
    $this->assertTrue($findings[0]['deprecated_events'][0]['critical']);
  }

  /**
   * Tests Drupal 11 event changes detection.
   */
  public function testDrupal11EventChangesDetection() {
    $code = <<<'EOT'
<?php
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class MyEventSubscriber implements EventSubscriberInterface {
  public static function getSubscribedEvents() {
    return [
      'kernel.view' => ['onView', 0],
      'kernel.response' => ['onResponse', 0],
    ];
  }
}
EOT;

    $visitor = new EventSubscriberVisitor();
    $this->traverser->addVisitor($visitor);
    
    $ast = $this->parser->parse($code);
    $this->traverser->traverse($ast);
    
    $findings = $visitor->getFindings();
    $this->assertNotEmpty($findings);
    
    $this->assertNotEmpty($findings[0]['drupal11_events']);
    $found_d11_changes = false;
    foreach ($findings[0]['drupal11_events'] as $event) {
      if ($event['event'] === 'kernel.view') {
        $found_d11_changes = true;
        $this->assertNotEmpty($event['changes']['changes']);
        break;
      }
    }
    $this->assertTrue($found_d11_changes, 'Should detect Drupal 11 event changes');
  }

}
