<?php

namespace Drupal\ai_vdb_provider_azure_ai_search;

use Drupal\Component\Utility\Crypt;
use Drupal\Core\Cache\CacheBackendInterface;
use GuzzleHttp\Exception\ClientException;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ConnectException;
use GuzzleHttp\Exception\RequestException;

/**
 * Extends Azure AI Search with extra calls.
 */
class AzureAiSearch {

  use StringTranslationTrait;

  const SUPPORTED_METHODS = ['GET', 'POST', 'PUT', 'DELETE'];

  /**
   * The Azure AI Search described indexes.
   *
   * @var array
   */
  private array $indexes = [];

  /**
   * The Azure AI Search api key.
   *
   * @var string
   */
  private string $apiKey = '';

  /**
   * Index name.
   *
   * @var string
   */
  private $indexName;

  /**
   * Index host URL.
   *
   * @var string
   */
  private $url;

  /**
   * API version.
   *
   * @var string
   */
  private $apiVersion;

  /**
   * Construct the Azure AI Search wrapper for the API.
   *
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The default cache bin.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerChannelFactory
   *   The logger factory.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The config factory.
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   The HTTP client.
   */
  public function __construct(
    protected CacheBackendInterface $cache,
    protected MessengerInterface $messenger,
    protected LoggerChannelFactoryInterface $loggerChannelFactory,
    protected ConfigFactoryInterface $configFactory,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ClientInterface $httpClient,
  ) {
  }

  /**
   * Get the client.
   *
   * @param string $api_key
   *   The AzureAiSearch client.
   *
   * @return \Drupal\ai_vdb_provider_azure_ai_search\AzureAiSearch
   *   The AzureAiSearch client.
   */
  public function getClient(string $api_key): AzureAiSearch {
    $config = $this->getConfig();
    $this->url = $config->get('url');
    $this->apiVersion = $config->get('api_version');
    $this->apiKey = $api_key;

    return $this;
  }

  /**
   * Helper method to get the client preconfigured for a specific index.
   *
   * @param string $index_name
   *   The index name.
   *
   * @return \Drupal\ai_vdb_provider_azure_ai_search\AzureAiSearch
   *   The Azure AI Search client.
   */
  protected function getClientForIndex(string $index_name): AzureAiSearch {
    $this->indexName = $index_name;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getConfig(): ImmutableConfig {
    return $this->configFactory->get('ai_vdb_provider_azure_ai_search.settings');
  }

  /**
   * Clear the indexes cache.
   */
  public function clearIndexesCache(): void {
    $this->indexes = [];
    $cid = 'azure_ai_search:' . Crypt::hashBase64($this->apiKey);
    $this->cache->delete($cid);
  }

  /**
   * Get all indexes available via the Azure AI Search API.
   *
   * @return array
   *   The available indexes.
   */
  public function listIndexes(): array {
    if ($this->indexes) {
      return $this->indexes;
    }

    $cid = 'azure_ai_search:' . Crypt::hashBase64($this->apiKey);
    if ($cache = $this->cache->get($cid)) {
      return $cache->data;
    }

    try {
      $response = $this->request('/indexes', 'GET', [
        'headers' => ['Content-Type' => 'application/json'],
      ]);

      if ($response->isSuccessful() && !empty($response->getBody())) {
        $indexes = $response->getBody()['value'];
        $this->cache->set($cid, $indexes);
        return $indexes;
      }
    }
    catch (\Exception $exception) {
      $this->messenger->addWarning($this->t('An exception occurred: @exception', [
        '@exception' => $exception->getMessage(),
      ]));
    }
    return [];
  }

  /**
   * Describe an index.
   *
   * @param string $index_name
   *   The index name.
   *
   * @return array
   *   The index described.
   */
  public function describeIndex(string $index_name): array {
    foreach ($this->listIndexes() as $index) {
      if ($index['name'] === $index_name) {
        return $index;
      }
    }
    return [];
  }

  /**
   * Gets all index stats.
   */
  public function getIndexStats(string $index_name): array {
    $response = $this->request("/indexes('$index_name')/search.stats");

    if ($response->isSuccessful() && !empty($response->getBody())) {
      $body = $response->getBody();
      return [
        'count' => $body['documentCount'],
        'size' => $body['storageSize'],
      ];
    }
    return [];
  }

  /**
   * Insert into the index.
   *
   * @param array $data
   *   The data.
   * @param string $index_name
   *   The index name.
   */
  public function insert(array $data, string $index_name): void {
    try {
      $index = $this->entityTypeManager->getStorage('search_api_index')->load($data['index_id']);
      $data['@search.action'] = 'mergeOrUpload';
      $data['id'] = str_replace([':', '/'], '_', $data['drupal_long_id']);

      $this->getClientForIndex($index_name)->request('/indexes/' . $index_name . '/docs/index', 'POST', [
        'json' => [
          'value' => [$data],
        ],
      ]);
    }
    catch (\Exception $exception) {
      $this->messenger->addWarning($this->t('An exception occurred while attempting to insert or update data in Azure AI Search: @exception', [
        '@exception' => $exception->getMessage(),
      ]));
      $this->loggerChannelFactory->get('ai_search')->warning('An exception occurred while attempting to insert or update data in Azure AI Search: @exception', [
        '@exception' => $exception->getMessage(),
      ]);
    }
  }

  /**
   * Delete items from the index.
   *
   * @param array $ids
   *   The ids.
   * @param string $index_name
   *   The index name.
   */
  public function delete(array $ids, string $index_name): void {
    $data = [];
    foreach ($ids as $id) {
      $data[] = [
        '@search.action' => 'delete',
        'id' => $id,
      ];
    }

    try {
      $this->getClientForIndex($index_name)->request('/indexes/' . $index_name . '/docs/index', 'POST', [
        'json' => [
          'value' => $data,
        ],
      ]);
    }
    catch (\Exception $exception) {
      $this->messenger->addWarning($this->t('An exception occurred while attempting to delete data in Azure AI Search: @exception', [
        '@exception' => $exception->getMessage(),
      ]));
      $this->loggerChannelFactory->get('ai_search')->warning('An exception occurred while attempting to delete data in Azure AI Search: @exception', [
        '@exception' => $exception->getMessage(),
      ]);
    }
  }

  /**
   * Delete all items.
   *
   * @param string $index_name
   *   The index name.
   */
  public function deleteAll(string $index_name): void {
    // @todo Add code for delete all.
  }

  /**
   * Look up and returns vectors by ID.
   *
   * @param array $ids
   *   The IDs to fetch.
   * @param string $index_name
   *   The index name.
   *
   * @return array
   *   The IDs.
   */
  public function fetch(array $ids, string $index_name): array {
    $filter = "drupal_entity_id eq '" . implode("' or drupal_entity_id eq '", $ids) . "'";

    $response = $this->request('/indexes/' . $index_name . '/docs/search', 'POST', [
      'json' => [
        'search' => '',
        "filter" => $filter,
      ],
    ]);

    if ($response->isSuccessful() && !empty($response->getBody()['value'])) {
      return $response->getBody()['value'];
    }
    return [];
  }

  /**
   * Query the vector index without providing vector values.
   *
   * @param string $index_name
   *   The index name.
   * @param array $filter
   *   The filters as a PHP array to be converted to OData format to match the
   *   expected format in the Azure AI Search documentation.
   * @param int $topK
   *   The number of results to return.
   * @param array $vector
   *   The vector array to search by. Leave as an empty array for no search.
   *
   * @return array
   *   The response.
   */
  public function query(string $index_name, array $filter = [], int $topK = 10, array $vector = []): array {
    $json = [
      'count' => TRUE,
      'select' => '*',
    ];

    if ($filter) {
      $filter_val = '';
      foreach ($filter as $field_name => $conditions) {
        foreach ($conditions as $operator => $value) {
          $filter_val .= $field_name . ' ' . str_replace('$', '', $operator) . ' \'' . $value . '\'';
        }
      }
      $json['filter'] = $filter_val;
    }

    if ($vector) {
      $json['vectorQueries'] = [
        [
          'vector' => $vector,
          'k' => $topK,
          'fields' => 'vector',
          'kind' => 'vector',
          'exhaustive' => TRUE,
        ],
      ];
    }

    $response = $this->getClientForIndex($index_name)->request('/indexes/' . $index_name . '/docs/search', 'POST', [
      'json' => $json,
    ]);

    $results = [];
    if ($response->isSuccessful() && $response->getBody()['value']) {
      foreach ($response->getBody()['value'] as $response_data) {
        $data = [];
        $data['score'] = $response_data['@search.score'];
        $data['metadata'] = $response_data;
        $results[] = $data;
      }
    }

    return $results;
  }

  /**
   * Create a complete URL ready for use.
   *
   * @param string $path
   *   The path to the endpoint.
   * @param array $http_options
   *   A list of HTTP options.
   *
   * @return string
   *   A complete endpoint URL
   */
  protected function buildUrl(string $path, array $http_options = []) : string {
    $http_options = [
      'api-version' => $this->apiVersion,
    ] + $http_options;

    return rtrim($this->url, '/') . '/' . ltrim($path, '/') . '?' . http_build_query($http_options);
  }

  /**
   * Validate if the HTTP method is valid.
   *
   * @param string $method
   *   The HTTP method.
   *
   * @throws \InvalidArgumentException
   *   An invalid method has been supplied.
   */
  protected function validateMethod(string $method): void {
    if (!in_array($method, self::SUPPORTED_METHODS, TRUE)) {
      throw new \InvalidArgumentException('The used method is not supported.');
    }
  }

  /**
   * Executes an API request on this client.
   *
   * @param string $path
   *   The path of the endpoint.
   * @param string $method
   *   The HTTP method to use.
   * @param array $options
   *   The options' data.
   *
   * @return \Drupal\ai_vdb_provider_azure_ai_search\ResponseData
   *   A response data object.
   *
   * @throws \InvalidArgumentException
   *   An invalid method has been supplied.
   */
  public function request(string $path, string $method = 'GET', array $options = []): ResponseData {
    $url = $this->buildUrl($path);
    $this->validateMethod(strtoupper($method));

    // When an API key is supplied, add it.
    if ($this->apiKey) {
      if (!isset($options['headers'])) {
        $options['headers'] = [];
      }
      $options['headers'] += ['api-key' => $this->apiKey];
    }

    try {
      $response = $this->httpClient->request($method, $url, $options);
      $response_data = ResponseData::fromResponse($response);
    }
    catch (ClientException | RequestException $e) {
      $status_code = $e->getResponse() ? $e->getResponse()->getStatusCode() : 500;

      // Catch & log all exceptions, except for 404's.
      switch ($status_code) {
        case 404:
          // We do not log 404's for the time being, as in some situations a
          // 404 is actually the desired result.
          // When trying to create a remote index for example, we need to know
          // if the index already exists or not. To do this, we need to send a
          // GET-request, and only actually proceed with creating the remote
          // index if the request returns a 404.
          // @todo Filter on request type. Log when not expected.
          break;

        default:
          // In case of a ClientException, try and log the exception in a
          // readable format.
          // See https://learn.microsoft.com/en-us/rest/api/searchservice/http-status-codes
          // for more info on specific status codes.
          // @todo Can be simplified.
          if ($e instanceof ClientException) {
            try {
              $e_contents = json_decode($e->getResponse()->getBody()->getContents(), TRUE);
              if ($e_contents) {
                if (array_key_exists('details', $e_contents['error'])) {
                  $this->loggerChannelFactory->get('ai_search')->error(implode(': ', $e_contents['error']['details'][0]));
                }
                elseif (array_key_exists('message', $e_contents['error'])) {
                  $this->loggerChannelFactory->get('ai_search')->error($e_contents['error']['message']);
                }
              }
              else {
                $this->loggerChannelFactory->get('ai_search')->error($e->getMessage());
              }
            }
            catch (\Error $x) {
              $this->loggerChannelFactory->get('ai_search')->error($x->getMessage());
            }
          }
          else {
            $this->loggerChannelFactory->get('ai_search')->error($e->getMessage());
          }
          break;
      }

      $response_data = new ResponseData();
      $response_data->setStatusCode($status_code);
    }
    catch (ConnectException $e) {
      $this->loggerChannelFactory->get('ai_search')->error($e->getMessage());

      $response_data = new ResponseData();
      $response_data->setStatusCode(400);
    }

    return $response_data;
  }

}
