<?php

namespace Drupal\ai_vdb_provider_azure_ai_search\Plugin\VdbProvider;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\ai\Attribute\AiVdbProvider;
use Drupal\ai\Base\AiVdbProviderClientBase;
use Drupal\ai\Enum\VdbSimilarityMetrics;
use Drupal\ai_vdb_provider_azure_ai_search\AzureAiSearch;
use Drupal\key\KeyRepositoryInterface;
use Drupal\search_api\Query\ConditionGroupInterface;
use Drupal\search_api\Query\QueryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Plugin implementation of the 'Azure AI Search' provider.
 */
#[AiVdbProvider(
  id: 'azure_ai_search',
  label: new TranslatableMarkup('Azure AI Search DB'),
)]
class AzureAiSearchProvider extends AiVdbProviderClientBase implements ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * The API key.
   *
   * @var string
   */
  protected string $apiKey = '';

  /**
   * Azure AI search service.
   *
   * @var \Drupal\ai_vdb_provider_azure_ai_search\AzureAiSearch
   */
  protected $azureAiSearch;

  /**
   * Constructs an override for the AiVdbClientBase class to add Milvus V2.
   *
   * @param string $pluginId
   *   Plugin ID.
   * @param mixed $pluginDefinition
   *   Plugin definition.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\key\KeyRepositoryInterface $keyRepository
   *   The key repository.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $eventDispatcher
   *   The event dispatcher.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger.
   * @param \Drupal\ai_vdb_provider_azure_ai_search\AzureAiSearch $azure_ai_search
   *   The AzureAiSearch API client.
   */
  public function __construct(
    protected string $pluginId,
    protected mixed $pluginDefinition,
    protected ConfigFactoryInterface $configFactory,
    protected KeyRepositoryInterface $keyRepository,
    protected EventDispatcherInterface $eventDispatcher,
    protected EntityFieldManagerInterface $entityFieldManager,
    protected MessengerInterface $messenger,
    protected AzureAiSearch $azure_ai_search,
  ) {
    parent::__construct(
      $this->pluginId,
      $this->pluginDefinition,
      $this->configFactory,
      $this->keyRepository,
      $this->eventDispatcher,
      $this->entityFieldManager,
      $this->messenger,
    );
    $this->azureAiSearch = $azure_ai_search;
  }

  /**
   * Load from dependency injection container.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): AiVdbProviderClientBase|static {
    return new static(
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('key.repository'),
      $container->get('event_dispatcher'),
      $container->get('entity_field.manager'),
      $container->get('messenger'),
      $container->get('azure_ai_search.api'),
    );
  }

  /**
   * Get the AzureAiSearch client.
   *
   * @return \Drupal\ai_vdb_provider_azure_ai_search\AzureAiSearch
   *   The AzureAiSearch client.
   */
  public function getClient($host = TRUE): AzureAiSearch {
    $key_name = $this->getConfig()->get('api_key');
    $key_value = $this->keyRepository->getKey($key_name)->getKeyValue();
    return $this->azureAiSearch->getClient($key_value);
  }

  /**
   * {@inheritdoc}
   */
  public function getConfig(): ImmutableConfig {
    return $this->configFactory->get('ai_vdb_provider_azure_ai_search.settings');
  }

  /**
   * {@inheritdoc}
   */
  public function getEditableConfig(): Config {
    return $this->configFactory->getEditable('ai_vdb_provider_azure_ai_search.settings');
  }

  /**
   * {@inheritdoc}
   */
  public function buildSettingsForm(
    array $form,
    FormStateInterface $form_state,
    array $configuration,
  ): array {
    // Don't load from cache here.
    $this->getClient()->clearIndexesCache();

    $form = parent::buildSettingsForm($form, $form_state, $configuration);

    // Add URL and API version fields.
    $form['url'] = [
      '#title' => $this->t('Url'),
      '#description' => $this->t('The URL to your Azure AI Search server.'),
      '#type' => 'url',
      '#required' => TRUE,
      '#default_value' => $configuration['database_settings']['url'] ?? '',
    ];

    $form['api_version'] = [
      '#title' => $this->t('API-version'),
      '#type' => 'select',
      '#options' => [
        '2020-06-30' => $this->t('2020-06-30 (Stable)'),
        '2020-06-30-Preview' => $this->t('2020-06-30 (Preview)'),
        '2021-04-30-Preview' => $this->t('2021-04-30 (Preview)'),
        '2023-10-01-preview' => $this->t('2023-10-01 (Preview)'),
        '2023-11-01' => $this->t('2023-11-01 (Stable)'),
        '2024-03-01-preview' => $this->t('2024-03-01 (Preview)'),
        '2024-05-01-preview' => $this->t('2024-05-01 (Preview)'),
      ],
      '#required' => TRUE,
      '#default_value' => $configuration['database_settings']['api_version'] ?? '2023-11-01',
      '#description' => $this->t('See <a href="@api-versions" target="_blank">the documentation</a> for the complete list of API-versions.', [
        '@api-versions' => Url::fromUri('https://docs.microsoft.com/en-us/rest/api/searchservice/search-service-api-versions')->toString(),
      ]),
    ];

    // Display all the key names available and let user choose the API key used
    // to connect to Azure AI Search.
    $keys = $this->keyRepository->getKeyNamesAsOptions();
    $api_keys = [];
    foreach ($keys as $key_name => $value) {
      $api_keys[$key_name] = $value;
    }
    $form['api_key'] = [
      '#title' => $this->t('API Key'),
      '#description' => $this->t('The URL to your Azure AI Search server.'),
      '#type' => 'select',
      '#required' => TRUE,
      '#default_value' => $configuration['database_settings']['api_key'] ?? '',
      '#options' => $api_keys,
    ];

    // Azure AI Search does not have the concept of collection. But we need a
    // value for the code to work. So we set it empty and hide the field from
    // the user.
    $form['collection']['#access'] = FALSE;
    $form['collection']['#value'] = '';

    unset($form['metric']);

    // Override the database name field.
    // Force the user to select from an index created via Azure Search UI.
    // This greatly simplifies what we need to handle in this module.
    unset($form['database_name']['#pattern']);
    $form['database_name']['#title'] = $this->t('Index Name');
    $form['database_name']['#description'] = $this->t('The index to use. Use the same name of the index as in Azure AI Search.');

    return $form;
  }

  /**
   * AJAX callback to update database field.
   */
  public function updateDatabaseField(array &$form, FormStateInterface $form_state) {
    return $form['database_name'];
  }

  /**
   * Get options for database name field.
   */
  protected function getIndexNames() {
    $indexes = $this->getClient()->listIndexes();
    $index_names = [];
    foreach ($indexes as $index) {
      $index_names[$index['name']] = $index['name'];
    }

    return $index_names;
  }

  /**
   * {@inheritdoc}
   */
  public function validateSettingsForm(array &$form, FormStateInterface $form_state): void {
    $database_settings = $form_state->getValue('database_settings');
    if (empty($database_settings['database_name'])) {
      $form_state->setErrorByName('backend_config][database_name', $this->t('Ensure that your Azure AI Search API key is correct and that you have created at least one Index in the Azure AI Search UI.'));
      return;
    }

    if (!$this->ping()) {
      $form_state->setError('backend_config][database_name', $this->t('Ensure that your Azure AI Search API key is correct and that you have created at least one Index in the Azure AI Search UI.'));
      return;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitSettingsForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->getEditableConfig();
    $form_values = $form_state->getValues();
    $config->set('url', $form_values['database_settings']['url']);
    $config->set('api_version', $form_values['database_settings']['api_version']);
    $config->set('api_key', $form_values['database_settings']['api_key']);
    $config->save();
  }

  /**
   * {@inheritdoc}
   */
  public function viewIndexSettings(array $database_settings): array {
    // Don't load from cache here.
    $this->getClient()->clearIndexesCache();

    $results = [];
    $results['ping'] = [
      'label' => $this->t('Ping'),
      'info' => $this->t('Able to reach Azure AI Search via their API.'),
      'status' => $this->ping() ? 'success' : 'error',
    ];

    if (!empty($database_settings['database_name'])) {
      $index_stats = $this->getClient()->getIndexStats($database_settings['database_name']);

      if (isset($index_stats['size'])) {
        $results['storage_size'] = [
          'label' => $this->t('Index storage size'),
          'info' => $index_stats['size'],
        ];
      }
      if (isset($index_stats['count'])) {
        $results['count'] = [
          'label' => $this->t('Total results count'),
          'info' => $index_stats['count'],
        ];
      }
    }
    return $results;
  }

  /**
   * {@inheritdoc}
   */
  public function ping(): bool {
    try {
      // If the API call fails, an exception will be thrown. If there are no
      // indexes, this will still succeed, which is fine.
      $this->getClient()->listIndexes();
      return TRUE;
    }
    catch (\Exception $e) {
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function isSetup(): bool {
    if ($this->getConfig()->get('api_key')) {
      return TRUE;
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function createCollection(
    string $collection_name,
    int $dimension,
    VdbSimilarityMetrics $metric_type = VdbSimilarityMetrics::CosineSimilarity,
    string $database = 'default',
  ): void {
    // Azure AI Search does not require the use of Collections.
  }

  /**
   * {@inheritdoc}
   */
  public function getCollections(string $database = 'default'): array {
    // Azure AI Search does not require the use of Collections.
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function dropCollection(
    string $collection_name,
    string $database = 'default',
  ): void {
    // Azure AI Search does not require the use of Collections.
  }

  /**
   * {@inheritdoc}
   */
  public function insertIntoCollection(
    string $collection_name,
    array $data,
    string $database = 'default',
  ): void {
    $this->getClient()->insert($data, $database);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteFromCollection(
    string $collection_name,
    array $ids,
    string $database = 'default',
  ): void {
    $this->getClient()->delete($ids, $database);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteAllItems(array $configuration, $datasource_id = NULL): void {
    // @todo Add code to delete all items.
  }

  /**
   * {@inheritdoc}
   */
  public function deleteItems(array $configuration, array $item_ids): void {
    $vdbIds = $this->getVdbIds(
      '',
      drupalIds: $item_ids,
      database: $configuration['database_settings']['database_name'],
    );
    if ($vdbIds) {
      $this->getClient()->delete(
        ids: $vdbIds,
        index_name: $configuration['database_settings']['database_name'],
      );
    }
  }

  /**
   * {@inheritdoc}
   */
  public function fetch(
    array $ids,
    string $database = 'default',
  ): array {
    // This fetches via the Drupal IDs.
    return $this->getClient()->fetch($ids, $database);
  }

  /**
   * {@inheritdoc}
   */
  public function getVdbIds(
    string $collection_name,
    array $drupalIds,
    string $database = 'default',
  ): array {
    // This gets the Azure AI Search IDs from the Drupal IDs.
    $data = $this->fetch(
      ids: $drupalIds,
      database: $database,
    );
    $ids = [];
    if (!empty($data)) {
      foreach ($data as $item) {
        $ids[] = $item['id'];
      }
    }
    return $ids;
  }

  /**
   * {@inheritdoc}
   */
  public function prepareFilters(QueryInterface $query): mixed {
    $index = $query->getIndex();
    $condition_group = $query->getConditionGroup();

    // Process filters, including handling nested groups.
    $filters = $this->processConditionGroup($index, $condition_group);

    // Combine all filters with $and if there are multiple conditions.
    if ($filters) {
      return count($filters) > 1 ? ['$and' => $filters] : $filters[0];
    }

    return [];
  }

  /**
   * Processes a condition group, including handling nested condition groups.
   */
  private function processConditionGroup($index, ConditionGroupInterface $condition_group): array {
    $filters = [];

    foreach ($condition_group->getConditions() as $condition) {
      // Check if the current condition is actually a nested ConditionGroup.
      if ($condition instanceof ConditionGroupInterface) {
        // Recursively process the nested ConditionGroup.
        $nestedFilters = $this->processConditionGroup($index, $condition);
        if ($nestedFilters) {
          // Add the nested filters as a grouped condition.
          $filters[] = ['$and' => $nestedFilters];
        }
      }
      else {
        $fieldData = $index->getField($condition->getField());
        $isMultiple = $fieldData ? $this->isMultiple($fieldData) : FALSE;
        $values = is_array($condition->getValue()) ? $condition->getValue() : [$condition->getValue()];
        $filter = [];

        // Handle multiple values fields.
        if ($isMultiple) {
          if (in_array($condition->getOperator(), ['=', 'IN'])) {
            // Use $in for Azure AI Search if the operator is '=' or 'IN'.
            $filter[$condition->getField()] = ['$in' => $values];
          }
          else {
            $this->messenger->addWarning('Azure AI Search does not support negative operator on multiple fields.');
          }
        }
        else {
          // Handle single value fields based on the operator.
          switch ($condition->getOperator()) {
            case '=':
              $filter[$condition->getField()] = ['$eq' => $values[0]];
              break;

            case '!=':
              $filter[$condition->getField()] = ['$ne' => $values[0]];
              break;

            case '>':
              $filter[$condition->getField()] = ['$gt' => $values[0]];
              break;

            case '>=':
              $filter[$condition->getField()] = ['$gte' => $values[0]];
              break;

            case '<':
              $filter[$condition->getField()] = ['$lt' => $values[0]];
              break;

            case '<=':
              $filter[$condition->getField()] = ['$lte' => $values[0]];
              break;

            case 'IN':
              $filter[$condition->getField()] = ['$in' => $values];
              break;

            case 'NOT IN':
              $filter[$condition->getField()] = ['$nin' => $values];
              break;

            default:
              // If the operator is not supported, log a warning.
              $this->messenger->addWarning('Operator @operator is not supported by Azure AI Search.', [
                '@operator' => $condition->getOperator(),
              ]);
              break;
          }
        }

        // Add the prepared filter to the list.
        if ($filter) {
          $filters[] = $filter;
        }
      }
    }

    return $filters;
  }

  /**
   * {@inheritdoc}
   */
  public function querySearch(
    string $collection_name,
    array $output_fields,
    mixed $filters = [],
    int $limit = 10,
    int $offset = 0,
    string $database = 'default',
  ): array {
    $matches = $this->getClient()->query(
      index_name: $database,
      filter: $filters,
      topK: $limit,
    );

    // Normalize the results to match what other VDB Providers return.
    $results = [];
    foreach ($matches as $match) {
      $results[] = $match['metadata'] + ['distance' => $match['score']];
    }
    return $results;
  }

  /**
   * {@inheritdoc}
   */
  public function vectorSearch(
    string $collection_name,
    array $vector_input,
    array $output_fields,
    mixed $filters = [],
    int $limit = 10,
    int $offset = 0,
    string $database = 'default',
  ): array {
    if (empty($vector_input)) {
      return [];
    }
    $matches = $this->getClient()->query(
      index_name: $database,
      filter: $filters,
      topK: $limit,
      vector: $vector_input,
    );

    // Normalize the results to match what other VDB Providers return.
    $results = [];
    foreach ($matches as $match) {
      $results[] = $match['metadata'] + ['distance' => $match['score']];
    }
    return $results;
  }

}
