<?php

namespace Drupal\ai_vdb_provider_mysql\Plugin\VdbProvider;

use Drupal\ai_search\EmbeddingStrategyInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Attribute\AiVdbProvider;
use Drupal\ai\Base\AiVdbProviderClientBase;
use Drupal\ai\Enum\VdbSimilarityMetrics;
use Drupal\ai_vdb_provider_mysql\VectorTableFactory;
use Drupal\key\KeyRepositoryInterface;
use Drupal\search_api\Entity\Index;
use Drupal\search_api\IndexInterface;
use Drupal\search_api\Query\QueryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Plugin implementation of the Mysql VDB provider.
 */
#[AiVdbProvider(
  id: 'mysql',
  label: new TranslatableMarkup('Mysql VDB Provider'),
)]
class MysqlProvider extends AiVdbProviderClientBase implements ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * Constructs an override for the AiVdbClientBase class to add Mysql VDB.
   *
   * @param string $pluginId
   *   Plugin ID.
   * @param mixed $pluginDefinition
   *   Plugin definition.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\key\KeyRepositoryInterface $keyRepository
   *   The key repository.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $eventDispatcher
   *   The event dispatcher.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger.
   * @param \Drupal\ai_vdb_provider_mysql\VectorTableFactory $vectorTableFactory
   *   The Mysql Vector Table factory.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   */
  public function __construct(
    protected string $pluginId,
    protected mixed $pluginDefinition,
    protected ConfigFactoryInterface $configFactory,
    protected KeyRepositoryInterface $keyRepository,
    protected EventDispatcherInterface $eventDispatcher,
    protected EntityFieldManagerInterface $entityFieldManager,
    protected MessengerInterface $messenger,
    protected VectorTableFactory $vectorTableFactory,
    protected Request $request,
  ) {
    parent::__construct(
      $this->pluginId,
      $this->pluginDefinition,
      $this->configFactory,
      $this->keyRepository,
      $this->eventDispatcher,
      $this->entityFieldManager,
      $this->messenger,
    );
  }

  /**
   * Load from dependency injection container.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): AiVdbProviderClientBase|static {
    return new static(
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('key.repository'),
      $container->get('event_dispatcher'),
      $container->get('entity_field.manager'),
      $container->get('messenger'),
      $container->get('mysql_vector.vector_table_factory'),
      $container->get('request_stack')->getCurrentRequest(),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getConfig(): ImmutableConfig {
    return $this->configFactory->get('ai_vdb_provider_mysql.settings');
  }

  /**
   * {@inheritdoc}
   */
  public function buildSettingsForm(
    array $form,
    FormStateInterface $form_state,
    array $configuration,
  ): array {
    $metric_distance = [
      VdbSimilarityMetrics::CosineSimilarity->value => $this->t('Cosine Similarity'),
    ];

    $form['metric'] = [
      '#type' => 'select',
      '#title' => $this->t('Similarity Metric'),
      '#options' => $metric_distance,
      '#required' => TRUE,
      '#default_value' => $configuration['database_settings']['metric'] ?? VdbSimilarityMetrics::CosineSimilarity->value,
      '#description' => $this->t('This mysql vdb implementation uses quantization + hamming distance first, and cosine similarity second.'),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitSettingsForm(array &$form, FormStateInterface $form_state): void {
    // Creating collections is handled by the backend server: addIndex().
  }

  /**
   * Get Vector Table client.
   *
   * This is needed for creating collections.
   *
   * @return \Drupal\ai_vdb_provider_mysql\VectorTableFactory
   *   The Mysql vector table.
   */
  public function getClient(): VectorTableFactory {
    return $this->vectorTableFactory;
  }

  /**
   * {@inheritdoc}
   */
  public function ping(): bool {
    // We might do a real ping to the database here or something?
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function isSetup(): bool {
    // We basically only have to check here whether Drupal uses a mysql
    // database that supports JSON fields.
    // Check if the database is MySQL.
    $database = \Drupal::database();
    $connection_options = $database->getConnectionOptions();

    if ($connection_options['driver'] !== 'mysql') {
      // Not a MySQL database.
      return FALSE;
    }

    // Check if the MySQL version supports JSON fields.
    // JSON support was added in MySQL 5.7.
    $version = $database->query('SELECT VERSION()')->fetchField();

    if (version_compare($version, '5.7.0', '<')) {
      // MySQL version does not support JSON fields.
      return FALSE;
    }

    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function validateSettingsForm(array &$form, FormStateInterface $form_state): void {
    // We remove all validations for collections since collections are
    // handled per index in our implementation.
    $database_settings = $form_state->getValue('database_settings');
    // Ensure that the user has been offered to configure the metrics, needed
    // if JS is disabled.
    if (!isset($database_settings['metric'])) {
      $form_state->setRebuild();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function viewIndexSettings(array $database_settings): array {
    $results = [];
    $results['ping'] = [
      'label' => $this->t('Ping'),
      'info' => $this->t('Pong!'),
      'status' => $this->ping() ? 'success' : 'error',
    ];

    return $results;
  }

  /**
   * {@inheritdoc}
   */
  public function getCollections(string $database = 'default'): array {
    // As database param we use the backend server id,
    // since all indexes there are the collections.
    $backend = \Drupal::entityTypeManager()
      ->getStorage('search_api_server')
      ->loadByProperties([
        'id' => $database
      ]);
    if (!$backend) {
      return [];
    }
    $backend = reset($backend);
    $query = \Drupal::entityTypeManager()
      ->getStorage('search_api_index')
      ->getQuery();
    $query->condition('server', $backend->id());
    $ids = $query->execute();

    if (!$ids) {
      return [];
    }
    return \Drupal::entityTypeManager()
      ->getStorage('search_api_index')
      ->loadMultiple($ids);
  }

  /**
   * {@inheritdoc}
   */
  public function createCollection(
    string $collection_name,
    int $dimension,
    VdbSimilarityMetrics $metric_type = VdbSimilarityMetrics::CosineSimilarity,
    string $database = 'default',
  ): void {
    $vectorTable = $this->getClient()->getVectorTable($collection_name, $dimension);
    $vectorTable->initialize();
    $state = \Drupal::state()->get('mysql_vector_tables');
    $collections = [];
    if ($state) {
      $collections = unserialize($state);
    }
    $newCollections = array_unique([...$collections, $collection_name]);
    \Drupal::state()->set('mysql_vector_tables', serialize($newCollections));
  }

  /**
   * {@inheritdoc}
   */
  public function deleteAllIndexItems(array $configuration, IndexInterface $index, $datasource_id = NULL): void {
    // Get the backend.
    $backend = \Drupal::entityTypeManager()->getStorage('search_api_server')
      ->loadByProperties([
        'id' => $index->getServerId()
      ]);
    if (!$backend) {
      return;
    }
    $backend = reset($backend);

    $dimensions = $backend->getBackendConfig()['embeddings_engine_configuration']['dimensions'];
    $table_name = $this->getClient()->getVectorTable($index->id(), $dimensions)->getVectorTableName();
    \Drupal::database()->truncate($table_name)->execute();
    $connection = \Drupal::database();
    $connection->delete('ai_vdb_provider_mysql')
      ->condition('collection', $index->id())
      ->execute();
  }

  /**
   * {@inheritdoc}
   */
  public function dropCollection(
    string $collection_name,
    string $database = 'default',
  ): void {
    // Get the default database connection.
    $database = \Drupal::database();

    // @todo we use the VectorTable class which needs dimensions to initialize.
    //   And we need it to initialize to find the correct table name.
    //   We can greatly simplify this by adapting the VectorTable class to our
    //   needs, and prevent an index and backend server lookup.

    // The index id is the same as our collection_name.
    $index = \Drupal::entityTypeManager()->getStorage('search_api_index')
      ->loadByProperties([
        'id' => $collection_name,
      ]);
    if (!$index) {
      return;
    }
    $index = reset($index);

    // Get the backend.
    $backend = \Drupal::entityTypeManager()->getStorage('search_api_server')
      ->loadByProperties([
        'id' => $index->getServerId()
      ]);
    if (!$backend) {
      return;
    }
    $backend = reset($backend);

    $dimensions = $backend->getBackendConfig()['embeddings_engine_configuration']['dimensions'];
    $table_name = $this->getClient()->getVectorTable($collection_name, $dimensions)->getVectorTableName();

    // Check if the table exists before attempting to drop it.
    if ($database->schema()->tableExists($table_name)) {
      // Drop the table.
      $database->schema()->dropTable($table_name);
      \Drupal::logger('ai_vdb_provider_mysql')->notice('Table @table_name has been dropped.', ['@table_name' => $table_name]);
      $database->delete('ai_vdb_provider_mysql')
        ->condition('collection', $collection_name)
        ->execute();
    }
    else {
      \Drupal::logger('ai_vdb_provider_mysql')->warning('Table @table_name does not exist.', ['@table_name' => $table_name]);
    }
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Exception
   */
  public function insertIntoCollection(
    string $collection_name,
    array $data,
    string $database = 'default',
  ): void {
    try {
      $id = $this->getClient()
        ->getVectorTable($collection_name, $this->getBackendConfig($collection_name)['embeddings_engine_configuration']['dimensions'])
        ->upsert($data['vector']);
      $database = \Drupal::database();
      $database->insert('ai_vdb_provider_mysql')
        ->fields([
          'entity' => $data['drupal_entity_id'],
          'vdb_id' => $id,
          'collection' => $collection_name,
        ])
        ->execute();
    }
    catch (\Exception $e) {
      \Drupal::logger('ai_vdb_provider_mysql')->error($e->getMessage());
    }
  }

  /**
   * {@inheritdoc}
   */
  public function deleteIndexItems(array $configuration, IndexInterface $index, array $item_ids): void {
    $vdb_ids = $this->getVdbIds($index->id(), $item_ids);
    $this->deleteFromCollection($index->id(), $vdb_ids);
    $connection = \Drupal::database();
    $connection->delete('ai_vdb_provider_mysql')
      ->condition('entity', $item_ids, 'IN')
      ->condition('collection', $index->id())
      ->execute();
  }

  /**
   * {@inheritdoc}
   */
  public function indexItems(
    array $configuration,
    IndexInterface $index,
    array $items,
    EmbeddingStrategyInterface $embedding_strategy,
  ): array {
    $successfulItemIds = [];
    $itemBase = [
      'metadata' => [
        'server_id' => $index->getServerId(),
        'index_id' => $index->id(),
      ],
    ];

    // Check if we need to delete some items first.
    $this->deleteIndexItems($configuration, $index, array_values(array_map(function ($item) {
      return $item->getId();
    }, $items)));

    /** @var \Drupal\search_api\Item\ItemInterface $item */
    foreach ($items as $item) {
      $embeddings = $embedding_strategy->getEmbedding(
        $configuration['embeddings_engine'],
        $configuration['chat_model'],
        $configuration['embedding_strategy_configuration'],
        $item->getFields(),
        $item,
        $index,
      );
      foreach ($embeddings as $embedding) {
        // Ensure consistent embedding structure as per
        // EmbeddingStrategyInterface.
        $this->validateRetrievedEmbedding($embedding);

        // Merge the base array structure with the individual chunk array
        // structure and add additional details.
        $embedding = array_merge_recursive($embedding, $itemBase);
        $data['drupal_long_id'] = $embedding['id'];
        $data['drupal_entity_id'] = $item->getId();
        $data['vector'] = $embedding['values'];
        foreach ($embedding['metadata'] as $key => $value) {
          $data[$key] = $value;
        }
        $this->insertIntoCollection(
          collection_name: $index->id(),
          data: $data,
        );
      }

      $successfulItemIds[] = $item->getId();
    }

    return $successfulItemIds;
  }

  /**
   * {@inheritdoc}
   */
  public function deleteFromCollection(
    string $collection_name,
    array $ids,
    string $database = 'default',
  ): void {
    try {
      foreach ($ids as $id) {
        $this->getClient()
          ->getVectorTable($collection_name, $this->getBackendConfig($collection_name)['embeddings_engine_configuration']['dimensions'])
          ->delete($id);
        $connection = \Drupal::database();
        $connection->delete('ai_vdb_provider_mysql')
          ->condition('vdb_id', $id)
          ->condition('collection', $collection_name)
          ->execute();
      }
    }
    catch (\Exception $e) {
      \Drupal::logger('ai_vdb_provider_mysql')->error($e->getMessage());
    }
  }

  /**
   * {@inheritdoc}
   */
  public function prepareFilters(QueryInterface $query): string {
    // Not supported at this time.
    return '';
  }

  /**
   * {@inheritdoc}
   */
  public function querySearch(
    string $collection_name,
    array $output_fields,
    mixed $filters = 'id not in [0]',
    int $limit = 10,
    int $offset = 0,
    string $database = 'default',
  ): array {
    // Not supported at this time.
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function vectorSearch(
    string $collection_name,
    array $vector_input,
    array $output_fields,
    mixed $filters = '',
    int $limit = 10,
    int $offset = 0,
    string $database = 'default',
  ): array {
    // The actual vector search:
    $data = $this->getClient()->getVectorTable($collection_name, $this->getBackendConfig($collection_name)['embeddings_engine_configuration']['dimensions'])->search($vector_input);
    $results = [];
    if (!empty($data)) {
      $connection = \Drupal::database();
      foreach ($data as $key => $row) {
        $drupal_entity_id = $connection->select('ai_vdb_provider_mysql', 'mvm')
          ->fields('mvm', ['entity'])
          ->condition('mvm.vdb_id', $row['id'])
          ->condition('mvm.collection', $collection_name)
          ->execute()->fetchField();
        if (!$drupal_entity_id) {
          // If for some reason a missing entry is found, delete it from the
          // index so it gets indexed again.
          $this->deleteFromCollection($collection_name, [$row['id']]);
          continue;
        }
        [$entity_type, $id_lang] = explode('/', str_replace('entity:', '', $drupal_entity_id));
        [$id, $lang] = explode(':', $id_lang);
        /** @var \Drupal\Core\Entity\ContentEntityInterface $entity */
        $entity = \Drupal::entityTypeManager()->getStorage($entity_type)->load($id);
        if ($entity->hasTranslation($lang)) {
          $entity = $entity->getTranslation($lang);
        }
        $results[$key]['drupal_entity_id'] = $drupal_entity_id ;
        $results[$key]['content'] = $entity->label();
        $results[$key]['distance'] = $row['similarity'];
      }
    }
    return $results;
  }

  /**
   * {@inheritdoc}
   */
  public function getVdbIds(
    string $collection_name,
    array $drupalIds,
    string $database = 'default',
  ): array {
    $connection = \Drupal::database();

    $query = $connection->select('ai_vdb_provider_mysql', 'mvm')
      ->fields('mvm', ['vdb_id'])
      ->condition('mvm.entity', $drupalIds, 'IN')
      ->condition('mvm.collection', $collection_name);
    $result = $query->execute()->fetchCol();

    return $result ?: [];
  }

  /**
   * {@inheritdoc}
   */
  public function addIndex(IndexInterface $index): void {
    $this->createCollection(
      $index->id(),
      $this->getBackendConfig($index->id())['embeddings_engine_configuration']['dimensions'],
    );
  }

  /**
   * {@inheritdoc}
   */
  public function removeIndex(IndexInterface $index): void {
    $this->dropCollection(
      $index->id(),
      $this->getBackendConfig($index->id())['embeddings_engine_configuration']['dimensions'],
    );
  }

  /**
   * Helper to get the backend config for an index.
   *
   * @param string $id
   *   The id of index.
   */
  private function getBackendConfig(string $id): array {
    // Get the index.
    $index = \Drupal::entityTypeManager()->getStorage('search_api_index')
      ->loadByProperties([
        'id' => $id,
      ]);
    if (!$index) {
      return [];
    }
    $index = reset($index);
    // Get the backend.
    $backend = \Drupal::entityTypeManager()->getStorage('search_api_server')
      ->loadByProperties([
        'id' => $index->getServerId()
      ]);
    if (empty($backend)) {
      return [];
    }
    $backend = reset($backend);
    return $backend->getBackendConfig() ?? [];
  }

}
