<?php

namespace Drupal\ai_vdb_provider_opensearch\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\ai\AiVdbProviderPluginManager;
use Drupal\key\KeyRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Opensearch DB config form.
 */
class OpensearchConfigForm extends ConfigFormBase {

  /**
   * The VDB PRovider service.
   *
   * @var \Drupal\ai\AiVdbProviderPluginManager
   */
  protected AiVdbProviderPluginManager $vdbProviderPluginManager;

  /**
   * The key repository.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected KeyRepositoryInterface $keyRepository;

  /**
   * Constructor of the Opensearch DB config form.
   *
   * @param \Drupal\ai\AiVdbProviderPluginManager $vdbProviderPluginManager
   *   The VDB Provider plugin manager.
   * @param \Drupal\key\KeyRepositoryInterface $keyRepository
   *   The key repository.
   */
  public function __construct(AiVdbProviderPluginManager $vdbProviderPluginManager, KeyRepositoryInterface $keyRepository) {
    $this->vdbProviderPluginManager = $vdbProviderPluginManager;
    $this->keyRepository = $keyRepository;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('ai.vdb_provider'),
      $container->get('key.repository'),
    );
  }

  /**
   * Config settings.
   */
  const CONFIG_NAME = 'ai_vdb_provider_opensearch.settings';

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'opensearch_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [
      static::CONFIG_NAME,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config(static::CONFIG_NAME);

    $form['server'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Server'),
      '#required' => TRUE,
      '#description' => $this->t('The OpenSearch server URL to connect to (e.g., "http://opensearch" for local DDEV installation or "https://my-opensearch-domain.region.es.amazonaws.com" for AWS OpenSearch).'),
      '#default_value' => $config->get('server'),
    ];

    $form['port'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Port'),
      '#description' => $this->t('The OpenSearch server port. Default is "9200" for standard OpenSearch installation. For AWS OpenSearch Service, use "443".'),
      '#default_value' => $config->get('port') ?: 9200,
    ];

    $form['api_key'] = [
      '#type' => 'key_select',
      '#title' => $this->t('Authentication'),
      '#description' => $this->t('The authentication credentials for OpenSearch. This should be stored as "username:password" for basic authentication. For AWS OpenSearch, you might need to use AWS Signature V4 authentication or API keys depending on your setup. Optional for local installations without security enabled.'),
      '#default_value' => $config->get('api_key'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $server = $form_state->getValue('server');
    if (!filter_var($server, FILTER_VALIDATE_URL)) {
      $form_state->setErrorByName('server', $this->t('The server must be a valid URL.'));
    }

    $port = $form_state->getValue('port');
    if (!empty($port) && !is_numeric($port)) {
      $form_state->setErrorByName('port', $this->t('The port must be a number.'));
    }

    // Test the connection.
    $opensearchConnector = $this->vdbProviderPluginManager->createInstance('opensearch');
    $key = $form_state->getValue('api_key');
    if (!empty($key)) {
      $key = $this->keyRepository->getKey($key)->getKeyValue();
    }
    $opensearchConnector->setCustomConfig([
      'server' => $server,
      'port' => $port,
      'api_key' => $key,
    ]);

    if (!$opensearchConnector->ping()) {
      $form_state->setErrorByName('server', $this->t('Could not connect to the server.'));
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config(static::CONFIG_NAME)
      ->set('server', rtrim($form_state->getValue('server'), '/'))
      ->set('port', $form_state->getValue('port'))
      ->set('api_key', $form_state->getValue('api_key'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
