<?php
namespace Drupal\ai_vdb_provider_sqlite\Form;

use Drupal\ai\AiVdbProviderPluginManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure SQLite vector DB config form.
 */
class SqliteConfigForm extends ConfigFormBase {

  /**
   * The VDB Provider plugin manager.
   *
   * @var \Drupal\ai\AiVdbProviderPluginManager
   */
  protected AiVdbProviderPluginManager $vdbProviderPluginManager;

  /**
   * Constructor of the SQLite vector DB config form.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   * The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   * The typed configuration manager.
   * @param \Drupal\ai\AiVdbProviderPluginManager $vdbProviderPluginManager
   * The VDB Provider plugin manager
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    TypedConfigManagerInterface $typedConfigManager,
    AiVdbProviderPluginManager $vdbProviderPluginManager
  ) {
    parent::__construct($configFactory, $typedConfigManager);
    $this->vdbProviderPluginManager = $vdbProviderPluginManager;
  }


  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('ai.vdb_provider')
    );
  }

  /**
   * Config settings.
   * Defines the configuration object name.
   */
  const CONFIG_NAME = 'ai_vdb_provider_sqlite.settings'; // Updated config name for SQLite

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'ai_vdb_provider_sqlite_settings'; // Updated form ID for SQLite
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [
      static::CONFIG_NAME,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config(static::CONFIG_NAME);

    // Field for the SQLite database file path.
    $form['db_path'] = [
      '#type' => 'textfield',
      '#required' => TRUE,
      '#title' => $this->t('Database Directory Path'),
      '#description' => $this->t('The absolute or relative path to the SQLite database path. Ensure the web server has write permissions to the directory if the file needs to be created or written to.').' '.$this->t("This location should not be downloadable by web server"),
      '#attributes' => ['placeholder' => 'eg. private://vdb'],
      '#default_value' => $config->get('db_path'),
    ];

    // Field for the optional SQLite extension file path.
    $extension_dir = ini_get('sqlite3.extension_dir');

    $form['ext_file'] = [
      '#type' => 'textfield',
      '#required' => TRUE,
      '#title' => $this->t('Extension File Path'),
      '#description' => $this->t('The filename to the SQLite vector extension file (e.g., vec0.so.).').' '.$this->t('This should under current sqlite.extension-dir ini setting <strong>@extdir</strong>', [
        '@extdir' => $extension_dir
      ]),
      '#default_value' => $config->get('ext_file'),
    ];


    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    // verify db_path
    $db_path = $form_state->getValue('db_path');
    $db_path = rtrim(trim($db_path), DIRECTORY_SEPARATOR);

    // Check if the provided path is actually a directory.
    if ($db_path === 'private://') {
      $form_state->setErrorByName('db_path', $this->t('You should create sub directory under %path', ['%path' => $db_path]));
      return;
    }
    if (!is_dir($db_path)) {
      if (file_exists($db_path)) {
        $form_state->setErrorByName('db_path', $this->t('The provided path is a file, but a directory is required: %path', ['%path' => $db_path]));
      } else {
        $form_state->setErrorByName('db_path', $this->t('The specified directory does not exist: %path', ['%path' => $db_path]));
      }
      return;
    }

    // Check if the directory is writable.
    if (!is_writable($db_path)) {
      $form_state->setErrorByName('db_path', $this->t('The specified directory is not writable: %path', ['%path' => $db_path]));
      return;
    }

    // test sqlite can be create
    $test_db_filename = 'verify-downloadable.sqlite.sql';
    $test_db_path = $db_path . DIRECTORY_SEPARATOR . $test_db_filename;
    $test_db_path = \Drupal::service('file_system')->realpath($test_db_path);
    $sqlite = null;

    try {
      $sqlite = new \SQLite3($test_db_path, SQLITE3_OPEN_READWRITE | SQLITE3_OPEN_CREATE);
    } catch (\Exception $e) {
      $form_state->setErrorByName('db_path', $this->t('Failed to create or open a test database file in the specified directory (%dir). Error: %error', [
        '%dir' => $db_path,
        '%error' => $e->getMessage(),
      ]));
      // No return here yet, proceed to finally block for cleanup.

    } finally {
      // Ensure the SQLite connection is closed if it was opened.
      if ($this->isFilePubliclyAccessible($test_db_path)) {
        $form_state->setErrorByName('db_path', $this->t('Security risk: The database file is publicly accessible. Please use a location not accessible from the web.'));
        return;
      }
    }

    // verify ext
    if ($sqlite && !empty($ext_file)) {
      try {
        // Check if the extension file itself exists and is readable.
        $sqlite_ext_dir = ini_get('sqlite3.extension_dir');
        if (!file_exists($sqlite_ext_dir.'/'.$ext_file)) {
           $form_state->setErrorByName('ext_file', $this->t('Extension file not found at the specified path: %path', ['%path' => $ext_file]));
        } elseif (!is_readable($sqlite_ext_dir.'/'.$ext_file)) {
           $form_state->setErrorByName('ext_file', $this->t('Extension file is not readable: %path', ['%path' => $ext_file]));
        } else {
          // Attempt to load the extension

          if (empty($sqlite_ext_dir)) {
            $extension_path = $ext_file;
          }
          elseif (strpos(realpath($ext_file), realpath($sqlite_ext_dir)) === 0) {
            $extension_path = basename($ext_file);
          }
          else {
            $extension_path = $ext_file;
          }
          $sqlite->loadExtension($extension_path);
        }

      } catch (\Exception $e) {
        $form_state->setErrorByName('ext_file', $this->t('Failed to load SQLite extension (%file): %error', [
          '%file' => $ext_file,
          '%error' => $e->getMessage()
        ]));
      }
    }

    if ($sqlite instanceof \SQLite3) {
      $sqlite->close();
    }
    if (!empty($test_db_path) && file_exists($test_db_path)) {
      @unlink($test_db_path);
    }
    if ($form_state->hasAnyErrors()) {
      return;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Save the configuration values.
    $this->config(static::CONFIG_NAME)
      ->set('db_path', $form_state->getValue('db_path'))
      ->set('ext_file', $form_state->getValue('ext_file'))
      // Removed Postgres specific settings
      ->save();

    parent::submitForm($form, $form_state);
  }

  private function isFilePubliclyAccessible($file_path) {
    $site_url = \Drupal::request()->getSchemeAndHttpHost();
    $drupal_root = \Drupal::root();

    if (strpos(realpath($file_path), realpath($drupal_root)) === 0) {
      $relative_path = str_replace(realpath($drupal_root), '', realpath($file_path));
      $relative_path = ltrim($relative_path, '/\\');
      $test_url = $site_url . '/' . $relative_path;

      try {
        /** @var \GuzzleHttp\ClientInterface $http_client */
        $http_client = \Drupal::httpClient();
        $response = $http_client->request('HEAD', $test_url, [
          'timeout' => 5,
          'connect_timeout' => 5,
          'allow_redirects' => true,
          'http_errors' => false,
        ]);

        return $response->getStatusCode() == 200;
      }
      catch (\Exception $e) {
        return false;
      }
    }

    return false;
  }

}