<?php

namespace Drupal\ai_vdb_provider_sqlite;

use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\ai_vdb_provider_sqlite\Exception\AddFieldIfNotExistsException;
use Drupal\ai_vdb_provider_sqlite\Exception\CreateCollectionException;
use Drupal\ai_vdb_provider_sqlite\Exception\DatabaseConnectionException;
use Drupal\ai_vdb_provider_sqlite\Exception\DeleteFromCollectionException;
use Drupal\ai_vdb_provider_sqlite\Exception\DropCollectionException;
use Drupal\ai_vdb_provider_sqlite\Exception\GetCollectionsException;
use Drupal\ai_vdb_provider_sqlite\Exception\InsertIntoCollectionException;
use Drupal\ai_vdb_provider_sqlite\Exception\QuerySearchException;
use Drupal\ai_vdb_provider_sqlite\Exception\VectorSearchException;
use SQLite3;

/**
 * Provides abstracted SQLite client to interface with vector extensions.
 */
class SQLiteVectorClient {

  protected const DATA_TYPE_MAPPING = [
    'integer' => 'INTEGER',
    'text' => 'TEXT',
    'date' => 'INTEGER',
    'decimal' => 'REAL',
    'string' => 'TEXT',
    'boolean' => 'INTEGER',
  ];

  /**
   * Get the SQLite database connection.
   *
   * @param string $filename
   * The path to the SQLite database file.
   *
   * @return SQLite3|FALSE
   * A connection to the SQLite database.
   *
   * @throws \Drupal\ai_vdb_provider_sqlite\Exception\DatabaseConnectionException
   */
  public function getConnection(string $db_file): SQLite3|FALSE {
    try {
      // SQLITE3_OPEN_CREATE: Create the database if it does not exist
      $connection = new SQLite3($db_file, SQLITE3_OPEN_READWRITE | SQLITE3_OPEN_CREATE);

      // Load SQLite extension from configuration
      $extension_file = \Drupal::config('ai_vdb_provider_sqlite.settings')->get('ext_file');
      if ($extension_file) {
        $connection->loadExtension($extension_file);
      }

      // Enable foreign keys
      // $connection->exec('PRAGMA foreign_keys = ON;');
    } catch (\Exception $e) {
      throw new DatabaseConnectionException(
      message: 'Cannot connect to SQLite database: ' . $e->getMessage(),
      );
    }
    if (!$connection) {
      // This path might be less likely due to exception handling above, but good practice
      throw new DatabaseConnectionException(
        message: 'Cannot connect to SQLite database using provided filename: ' . $db_file,
      );
    }
    return $connection;
  }

  /**
   * {@inheritdoc}
   */
  public function ping(SQLite3 $connection): bool {
    // Simple query to check if connection is alive
    $result = $connection->querySingle('SELECT 1');
    return $result === 1;
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\ai_vdb_provider_sqlite\Exception\GetCollectionsException
   */
  public function getCollections(SQLite3 $connection): array {
    // Query sqlite_master, exclude sqlite internal tables, vector indexes, and relation tables
    // Adjust the NOT LIKE patterns if your naming convention differs
    $query = "SELECT name FROM sqlite_master
          WHERE type='table'
          AND name NOT LIKE 'sqlite_%'
          AND name NOT LIKE '%_idx'
          AND name NOT LIKE '%__%'"; // Exclude relation tables like collection__field

    $result = $connection->query($query);
    if (!$result) {
      throw new GetCollectionsException(message: 'Failed to query tables: ' . $connection->lastErrorMsg());
    }

    $tables = [];
    while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
      $tables[] = $row['name'];
    }
    return $tables;
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\ai_vdb_provider_sqlite\Exception\CreateCollectionException
   */
  public function createCollection(
    string  $collection_name,
    int     $dimension,
    SQLite3   $connection,
  ): void {
    $escaped_collection_name = $this->escapeIdentifierForSql($collection_name);

    // 1. Create the main table for metadata
    $create_main_table_sql = "CREATE VIRTUAL TABLE IF NOT EXISTS {$escaped_collection_name} USING vec0 (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      content TEXT,
      drupal_entity_id TEXT,
      drupal_long_id TEXT,
      server_id TEXT,
      index_id TEXT,
      embedding float[{$dimension}]
    );";

    if (!$connection->exec($create_main_table_sql)) {
      throw new CreateCollectionException(message: "Failed to create metadata table '{$collection_name}': " . $connection->lastErrorMsg());
    }
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\ai_vdb_provider_sqlite\Exception\DropCollectionException
   */
  public function dropCollection(
  string  $collection_name,
  SQLite3   $connection,
  ): void {
    $escaped_collection_name = $this->escapeIdentifierForSql($collection_name);

    // Drop the main table
    if (!$connection->exec("DROP TABLE IF EXISTS {$escaped_collection_name};")) {
      throw new DropCollectionException(message: "Failed to drop metadata table '{$collection_name}': " . $connection->lastErrorMsg());
    }
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\ai_vdb_provider_sqlite\Exception\InsertIntoCollectionException
   */
  public function insertIntoCollection(
  string   $collection_name,
  array   $drupal_entity_id,
  array   $drupal_long_id,
  array   $content,
  array    $vector, // Contains ['value' => [float, float, ...]]
  array    $server_id,
  array     $index_id,
  array    $extra_fields, // ['field_name' => ['value' => mixed, 'is_multiple' => bool, 'type' => string]]
  SQLite3  $connection
  ): void {
    $vector_string = $this->prepareVectorArrayForSql(
      vector: $vector['value'],
      connection: $connection,
    );
    $escaped_collection_name = $this->escapeIdentifierForSql(
      identifier_to_escape: $collection_name,
      connection: $connection,
    );

    // Prepare columns and values for extra fields.
    $extra_fields_columns = '';
    $extra_fields_values = '';
    $extra_fields_params = [];

    $relation_queries = [];

    foreach ($extra_fields as $field_name => $field_data) {
      if ($field_data['is_multiple']) {
        if ($relation_query = $this->prepareRelationQuery($collection_name, $field_name, $field_data, $connection)) {
          $relation_queries[] = $relation_query;
        }
      } else {
        $extra_fields_columns .= ", {$field_name}";
        $extra_fields_values .= ", ?";
        $extra_fields_params[] = $field_data['value'];
      }
    }

    $main_query = "INSERT INTO {$escaped_collection_name} (content, drupal_entity_id, drupal_long_id, server_id, index_id, embedding{$extra_fields_columns}) VALUES (?, ?, ?, ?, ?, {$vector_string}{$extra_fields_values});";

    $stmt = $connection->prepare($main_query);

    // Bind parameters
    $stmt->bindValue(1, $content['value'], SQLITE3_TEXT);
    $stmt->bindValue(2, $drupal_entity_id['value'], SQLITE3_TEXT);
    $stmt->bindValue(3, $drupal_long_id['value'], SQLITE3_TEXT);
    $stmt->bindValue(4, $server_id['value'], SQLITE3_TEXT);
    $stmt->bindValue(5, $index_id['value'], SQLITE3_TEXT);

    // Bind extra fields
    $param_index = 6;
    foreach ($extra_fields_params as $param) {
      $stmt->bindValue($param_index, $param, SQLITE3_TEXT);
      $param_index++;
    }

    $result = $stmt->execute();
    if (!$result) {
      throw new InsertIntoCollectionException(message: $connection->lastErrorMsg());
    }

    foreach ($relation_queries as $relation_query) {
      $result = $connection->query($relation_query);
      if (!$result) {
        throw new InsertIntoCollectionException(message: $connection->lastErrorMsg());
      }
    }

  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\ai_vdb_provider_sqlite\Exception\DeleteFromCollectionException
   */
  public function deleteFromCollection(
  string   $collection_name,
  array    $ids, // Drupal entity IDs
  SQLite3  $connection,
  ): void {
    if (empty($ids)) {
      return;
    }
    $escaped_collection_name = $this->escapeIdentifierForSql(
      identifier_to_escape: $collection_name,
      connection: $connection,
    );
    $prepared_ids = $this->prepareStringArrayForSql(items: $ids, connection: $connection);
    $result = $connection->query(
      "DELETE FROM {$escaped_collection_name} WHERE drupal_entity_id IN {$prepared_ids};"
    );
    if (!$result) {
      throw new DeleteFromCollectionException(message: $connection->lastErrorMsg());
    }
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\ai_vdb_provider_sqlite\Exception\QuerySearchException
   */
  public function querySearch(
  string   $collection_name,
  array    $output_fields, // Array of field names
  string   $filters, // Pre-constructed WHERE clause string (USE WITH CAUTION or adapt to use parameters)
  int    $limit,
  int    $offset,
  SQLite3  $connection,
  ): array {
    $escaped_collection_name = $this->escapeIdentifierForSql(
      identifier_to_escape: $collection_name,
      connection: $connection,
    );
    $prepared_output_fields = $this->prepareFieldArrayForSql(fields: $output_fields, connection: $connection, collection_name: $collection_name);
    if (empty($filters)) {
      $query = "SELECT {$prepared_output_fields} FROM {$escaped_collection_name} LIMIT {$limit} OFFSET {$offset};";
    }
    else {
      $query = "SELECT {$prepared_output_fields} FROM {$escaped_collection_name} {$filters} LIMIT {$limit} OFFSET {$offset};";
    }

    $result = $connection->query($query);
    if (!$result) {
      throw new QuerySearchException(message: $connection->lastErrorMsg());
    }

    $rows = [];
    while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
      $rows[] = $row;
    }

    return $rows;
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\ai_vdb_provider_sqlite\Exception\VectorSearchException
   */
  public function vectorSearch(
    string       $collection_name,
    array        $vector_input, // The query vector [float, float, ...]
    array        $output_fields, // Array of field names from the main table
    string       $filters, // Pre-constructed WHERE clause for the *main* table (USE WITH CAUTION)
    int          $limit,
    int          $offset,
    SQLite3      $connection
  ): array {
    $escaped_collection_name = $this->escapeIdentifierForSql(
      identifier_to_escape: $collection_name,
      connection: $connection,
    );
    $prepared_output_fields = $this->prepareFieldArrayForSql(fields: $output_fields, connection: $connection, collection_name: $collection_name);
    $vector_string = $this->prepareVectorArrayForSql(vector: $vector_input, connection: $connection);

    // Escape the output fields.
    $escaped_outfield_fields = array_map(
      callback: function ($field) use ($connection) {
        return $this->escapeIdentifierForSql(identifier_to_escape: $field, connection: $connection);
      },
      array: $output_fields
    );
    $outfield_fields = implode(',', $escaped_outfield_fields);

    // k = ? is temp solution from https://github.com/asg017/sqlite-vec/issues/116
    if (empty($filters)) {
      // $query = "SELECT {$outfield_fields} FROM {$escaped_collection_name} WHERE embedding MATCH {$vector_string} ORDER BY distance LIMIT {$limit};";
      $query = "SELECT {$outfield_fields} FROM {$escaped_collection_name} WHERE embedding MATCH {$vector_string} and k = {$limit} ORDER BY distance;";
    }
    else {
      // $query = "SELECT {$outfield_fields} FROM {$escaped_collection_name} WHERE embedding MATCH {$vector_string} {$filters} ORDER BY distance LIMIT {$limit};";
      $query = "SELECT {$outfield_fields} FROM {$escaped_collection_name} WHERE embedding MATCH {$vector_string} {$filters} AND k = {$limit} ORDER BY distance;";
    }

    $result = $connection->query($query);
    if (!$result) {
      throw new VectorSearchException(message: $connection->lastErrorMsg());
    }

    $rows = [];
    while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
      $rows[] = $row;
    }

    return $rows;
  }


  /**
   * Transform an array of field identifier strings for use in a SQL statement.
   * Adds table alias prefix if provided. Escapes identifiers.
   *
   * @param array $fields
   * Field array.
   * @param SQLite3 $connection (Kept for consistency, not strictly needed for current escaping)
   * The SQLite connection.
   * @param string|null $collection_name
   * Optional table alias to prefix fields (e.g., 'm').
   *
   * @return string
   * Array formatted as a field string. Eg: 'm."id", m."drupal_entity_id"'
   */
  public function prepareFieldArrayForSql(array $fields, SQLite3 $connection, ?string $collection_name = NULL): string {
    $array_formatted_as_string = '';
    $last_element = end(array: $fields);
    foreach ($fields as $field) {
      if ($collection_name) {
        $array_formatted_as_string .= $this->escapeIdentifierForSql(identifier_to_escape: $collection_name, connection: $connection) . '.';
      }
      if ($field === $last_element) {
        $array_formatted_as_string .=
          $this->escapeIdentifierForSql(identifier_to_escape: $field, connection: $connection) . '';
        break;
      }
      $array_formatted_as_string .=
        $this->escapeIdentifierForSql(identifier_to_escape: $field, connection: $connection) . ',';
    }
    return $array_formatted_as_string;
  }

  /**
   * Transform an array of non-string data to string for use in a SQL statement.
   *
   * @param array $items
   *   An array of string items.
   *
   * @return string
   *   Array formatted as a string for SQL.
   *   Eg: "('first item', 'second item', 'third item')"
   */
  public function prepareArrayForSql(array $items): string {
    return '(' . implode(separator: ',', array: $items) . ')';
  }

  /**
   * Transform an array of vectors to string for use in SQLite MATCH or INSERT.
   *
   * @param array $vector Vector array (normally an array of floats).
   *
   * @return string Array formatted as a string. Eg: '[1.22424,-2.12312,-1.34654]'
   */
  public function prepareVectorArrayForSql(array $vector, SQLite3 $connection): string {
    $json_string = json_encode($vector, JSON_NUMERIC_CHECK);
    return $this->escapeStringForSql(string_to_escape: $json_string, connection: $connection);
  }

  /**
   * Transform an array of strings to string for use in a SQL statement.
   *
   * @param array $items
   *   An array of string items.
   * @param SQLite3 $connection
   *   The SQLite connection.
   *
   * @return string
   *   Array of strings formatted as a string for SQL.
   *   Eg: "('first item', 'second item', 'third item')"
   *
   * @throws \Drupal\ai_vdb_provider_sqlite\Exception\EscapeStringException
   */
  public function prepareStringArrayForSql(array $items, SQLite3 $connection): string {
    $escaped_strings = [];
    foreach ($items as $item) {
      $escaped_strings[] = $this->escapeStringForSql(string_to_escape: $item, connection: $connection);
    }
    return '(' . implode(separator: ',', array: $escaped_strings) . ')';
  }


  /**
   * Escape a string literal for use in a SQLite SQL statement.
   * NOTE: Prefer using prepared statements with bindValue instead of this.
   *
   * @param string $string_to_escape The string to escape.
   * @param SQLite3 $connection
   *
   * @return string A string containing the escaped data, suitable for direct inclusion in SQL (e.g., 'string''s value').
   */
  private function escapeStringForSql(string $string_to_escape, SQLite3 $connection): string {
    return "'" . SQLite3::escapeString($string_to_escape) . "'";
  }

  /**
   * Escape a string identifier (table/column name) for use in a SQLite SQL statement.
   * SQLite uses double quotes for identifiers.
   *
   * @param string $identifier_to_escape The string identifier to escape.
   * @param SQLite3 $connection
   *
   * @return string A string containing the escaped identifier. Eg: "my-table"
   */
  public function escapeIdentifierForSql(string $identifier_to_escape, SQLite3 $connection = NULL): string {
    return '"' . str_replace('"', '""', $identifier_to_escape) . '"';
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\ai_vdb_provider_sqlite\Exception\AddFieldIfNotExistsException
   */
  public function updateFields($fields, string $collection_name, SQLite3 $connection): void {
    foreach ($fields as $field) {
      $field_data_definition = $field->getDataDefinition();

      // Make assumption of basic data type if we can't get more info.
      if (!method_exists($field_data_definition, 'getFieldDefinition')) {
        $this->addFieldIfNotExists(FALSE, 'string', $field->getFieldIdentifier(), $collection_name, $connection);
        continue;
      }
      $isMultiple = TRUE;

      $field_definition = $field_data_definition->getFieldDefinition();
      if ($field_definition instanceof BaseFieldDefinition) {
        $field_cardinality = $field_definition->getCardinality();
      }
      else {
        $field_cardinality =
          $field_definition->get('fieldStorage')->getCardinality();
      }
      if ($field_cardinality === 1) {
        $isMultiple = FALSE;
      }
      $this->addFieldIfNotExists($isMultiple, $field->getType(), $field->getFieldIdentifier(), $collection_name, $connection);
    }
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\ai_vdb_provider_sqlite\Exception\AddFieldIfNotExistsException
   */
  protected function addFieldIfNotExists(bool $isMultiple, string $drupal_data_type, string $name, string $collection_name, SQLite3 $connection): void {
    $escaped_collection_name = $this->escapeIdentifierForSql(
      identifier_to_escape: $collection_name,
      connection: $connection,
    );
    $sqlite_type = self::DATA_TYPE_MAPPING[$drupal_data_type];
    $escaped_field_name = $this->escapeIdentifierForSql($name, $connection);

    // If isMultiple is true, create a new relationship table
    if ($isMultiple) {
      $relation_table = $this->getRelationTableName($collection_name, $name, $connection);
      $create_relation_table = "CREATE TABLE IF NOT EXISTS {$relation_table} (id INTEGER PRIMARY KEY AUTOINCREMENT, value {$sqlite_type} NOT NULL, chunk_id INTEGER NOT NULL, FOREIGN KEY(chunk_id) REFERENCES {$escaped_collection_name}(id) ON DELETE CASCADE);";
      $result = $connection->query($create_relation_table);
      if (!$result) {
        throw new AddFieldIfNotExistsException(message: $connection->lastErrorMsg());
      }
    } else {
      $query = "ALTER TABLE {$escaped_collection_name} ADD COLUMN {$escaped_field_name} {$sqlite_type};";
      $result = $connection->query($query);
      if (!$result) {
        throw new AddFieldIfNotExistsException(message: $connection->lastErrorMsg());
      }
    }

  }

  protected function prepareRelationQuery($collection_name, $field_name, $field_data, $connection) {
    $query = '';

    // SQLite uses the last_insert_rowid() function instead of currval sequence
    $last_insert_id = "last_insert_rowid()";

    // Prepare entries for relation table.
    $relation_table_fields = [];
    $escaped_relation_table_name = $this->getRelationTableName($collection_name, $field_name, $connection);
    if (!is_array($field_data['value'])) {
      $field_data['value'] = [$field_data['value']];
    }
    foreach ($field_data['value'] as $value) {
      if (empty($value)) {
        continue;
      }
      $relation_table_fields[$escaped_relation_table_name][] = $value;
    }

    foreach ($relation_table_fields as $escaped_relation_table_name => $field_values) {
      $query .= "INSERT INTO {$escaped_relation_table_name} (value, chunk_id) values ";
      $last_value = end($field_values);
      foreach ($field_values as $field_value) {
        $query .= "({$field_value}, {$last_insert_id})";
        if ($field_value === $last_value) {
          $query .= ';';
        }
        else {
          $query .= ",";
        }
      }
    }
    return $query;
  }

  /**
   * Gets the escaped name for a relation table.
   *
   * @param string $collection_name Main collection name.
   * @param string $field_name    Field name requiring relation table.
   *
   * @return string The escaped relation table name (e.g., "collection__field").
   */
  public function getRelationTableName(string $collection_name, string $field_name): string {
    // Use double underscore as separator (common convention)
    return $this->escapeIdentifierForSql("{$collection_name}__{$field_name}", );
  }

}