<?php

namespace Drupal\aidmi\Service;

use GeminiAPI\Client;
use GeminiAPI\Enums\MimeType;
use GeminiAPI\Resources\Parts\ImagePart;
use GeminiAPI\Resources\Parts\TextPart;
use Drupal\file\Entity\File;  // Correct namespace for Drupal File Entity
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Component\Utility\Html;

class GeminiAiService {

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The API client for Google Gemini.
   *
   * @var \GeminiAPI\Client
   */
  protected $client;

  /**
   * The API instructions.
   *
   * @var string
   */
  protected $apiInstructions;

  public function __construct(ConfigFactoryInterface $configFactory) {
    $this->configFactory = $configFactory;
    // Retrieve the configuration for the module
    $config = $configFactory->get('aidmi.settings');

    // Store the API key and instructions in the service
    $apiKey = $config->get('api_key');
    if (empty($apiKey)) {
      $apiKey = '0';
    }
    $this->client = new Client($apiKey);
    $this->apiInstructions = $config->get('api_instructions');
  }

  public function analyzeImage(int $fid): string {
    try {
      $file = File::load($fid);
      if (!$file) {
        throw new \Exception($this->t('File not found.'));
      }
  
      $imagePath = $file->getFileUri();
      $imageContent = file_get_contents($imagePath);
      if (!$imageContent) {
        throw new \Exception($this->t('Unable to open image file.'));
      }
  
      $response = $this->client->geminiProFlash1_5()->generateContent(
          new TextPart($this->apiInstructions),
          new ImagePart(
              MimeType::IMAGE_JPEG,
              base64_encode($imageContent)
          )
      );
  
      // Return the description from Gemini API
      $output = addslashes($response->text());
      return $output;
    } catch (\Exception $e) {
        return 'Error: ' . $e->getMessage();
    }
  }
  
}
