<?php

namespace Drupal\aidmi\Service;

use GeminiAPI\Client;
use GeminiAPI\GenerationConfig;
use GeminiAPI\Resources\Parts\PartInterface;
use GeminiAPI\Enums\MimeType;
use GeminiAPI\Resources\Parts\ImagePart;
use GeminiAPI\Resources\Parts\TextPart;
use Drupal\file\Entity\File;  // Correct namespace for Drupal File Entity
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Component\Utility\Html;

class GeminiAiService {

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The API client for Google Gemini.
   *
   * @var \GeminiAPI\Client
   */
  protected $client;

  /**
   * The API instructions.
   *
   * @var string
   */
  protected $apiInstructions;

  /**
   * The AIDmi controller.
   *
   * @var \Drupal\aidmi\Controller\AidmiController
   */
  protected $aidmiController;
  

  public function __construct(ConfigFactoryInterface $configFactory) {
    $this->configFactory = $configFactory;
    // Retrieve the configuration for the module.
    $config = $configFactory->get('aidmi.settings');
    // Get the input method.
    $apiInputMethod = $config->get('api_input_method');

    // Initialize the API key variable.
    $apiKey = '';

    // Check if the API key is to be retrieved from the file.
    if ($apiInputMethod === 'file_path') {
      // Get the API key file path from the configuration.
      $apiKeyFilePath = $config->get('api_key_file_path');
      $siteRoot = \Drupal::root(); // Get the site root.
      $fullFilePath = $siteRoot . '/' . ltrim($apiKeyFilePath, '/'); // Full file path.
      // Check if the file exists and is readable.
      if (file_exists($fullFilePath) && is_readable($fullFilePath)) {
        // Read the API key from the file.
        $apiKey = trim(file_get_contents($fullFilePath));
      } else {
        // Log an error and default the API key to an empty string or a default value.
        \Drupal::logger('aidmi')->error('API key file not found or not readable at @path', ['@path' => $fullFilePath]);
        $apiKey = '0'; // Fallback if file is not found or readable.
      }
    } else {
      // Get the API key from the configuration if the input method is 'api_key'.
      $apiKey = $config->get('api_key');
      
      // Default to '0' if the API key is empty.
      if (empty($apiKey)) {
        $apiKey = '0';
      }
    }
        
    $this->client = new Client($apiKey);
    $this->apiInstructions = $config->get('api_instructions');
  }

  public function analyzeImage(int $fid): string {
    try {
      $file = File::load($fid);
      if (!$file) {
        throw new \Exception($this->t('File not found.'));
      }
  
      $imagePath = $file->getFileUri();
      $imageContent = file_get_contents($imagePath);
      if (!$imageContent) {
        throw new \Exception($this->t('Unable to open image file.'));
      }
  
      $response = $this->client->geminiProFlash1_5()->generateContent(
          new TextPart($this->apiInstructions),
          new ImagePart(
              MimeType::IMAGE_JPEG,
              base64_encode($imageContent)
          )
      );
  
      // Return the description from Gemini API
      $output = $response->text();
      return $output;
    } catch (\Exception $e) {
        return 'Error: ' . $e->getMessage();
    }
  }
  
  public function analyzeContent(string $content, string $imagesJSON): string {
    try {
      // Decode JSON string into PHP array
      $jsonArray = json_decode($imagesJSON, true);
            
      $generateContent[] = new TextPart('Response can only be pure JSON, do not provide null value if something is empty, just keep it "" blank. Example: [{"images": [{"data-entity-uuid": "", "src": "", "before_alt": "", "recommendation": "", "decorative_image" = ""}]}, {"message": "error or other message if needed"}]. Review this content and describe the image. Actual images from the content are attached and referenced by data-entity-uuid.  Try to provide some text in context for an accessibilty alt text. For graphs, make sure the description provides a detail for the reader to understand and include some relevant information perceived from it. For individuals, provide a short description. Try to provide short details based on context of where image is used in text. Provide a json output with each image by data-entity-uuid, image src, the before alt text if available, and your recommendation. If there is a problem, respond with only an error JSON. Provide a true or false for decorative image where just an alt tag alone is decorative, but alt="" is false.');
      $generateContent[] = new TextPart($content);

      // Loop through the array of files in content.
      foreach ($jsonArray as $item) {
        // Get file and contents.
        $file = $this->getFileIdByUuid($item['data-entity-uuid']);
        $imagePath = $file->getFileUri();
        $imageContent = file_get_contents($imagePath);
        if (!$imageContent) {
          throw new \Exception($this->t('Unable to open image file.'));
        }
        
        // Attach images for Gemini AI review.
        $generateContent[] = new TextPart("data-entity-uuid: ". $item['data-entity-uuid']);
        $generateContent[] = new ImagePart(
              MimeType::IMAGE_JPEG,
              base64_encode($imageContent)
        );
      }

      // Create the response.
      $response = $this->client
        ->withRequestHeaders([
          'response_mime_type' => 'application/json'
        ])->geminiProFlash1_5()
        ->generateContent(...$generateContent);

      // Convert to full JSON and return the description from Gemini API.
      $output = $response->text();
      $output = str_replace("```json","[", $output);
      $output = str_replace("```","]", $output);

      return $output;
    } catch (\Exception $e) {
        return 'Error: ' . $e->getMessage();
    }
  }
  
  public function getFileIdByUuid(string $uuid) {
    // Query for the file entity using the UUID.
    $file = \Drupal::entityTypeManager()->getStorage('file')->loadByProperties(['uuid' => $uuid]);

    if ($file) {
      $file = reset($file);  // Get the first result.
      return $file;
    } else {
      throw new \Exception($this->t('File not found.'));
    }
  }
}
