<?php
namespace Drupal\aidmi\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for your module.
 */
class AidmiSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['aidmi.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'aidmi_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Get the configuration.
    $config = $this->config('aidmi.settings');

    // Get the selected API input method.
    $api_input_method = $config->get('api_input_method');

    // Get the saved file path.
    $api_key_file_path = $config->get('api_key_file_path');

    // Initialize the message for file existence check.
    $file_status_message = '';

    $site_root = \Drupal::root();
    
    // Check if the file input method is selected and if the file exists.
    if ($api_input_method === 'file_path' && !empty($api_key_file_path)) {
      $full_file_path = $site_root . '/' . ltrim($api_key_file_path, '/'); // Ensure no double slashes.
      if (file_exists($full_file_path)) {
        $file_status_message = $this->t('File found: @path', ['@path' => $full_file_path]);
        \Drupal::messenger()->addStatus($file_status_message); // Add as a status message.
      }
      else {
        $file_status_message = $this->t('File not found at the provided path: @path', ['@path' => $full_file_path]);
        \Drupal::messenger()->addError($file_status_message); // Add as an error message.
      }
    }

    // Radio buttons for API selection
    $form['selected_api'] = [
      '#type' => 'radios',
      '#title' => $this->t('Select API'),
      '#options' => [
        'gemini' => $this->t('Google Gemini'),
      ],
      '#default_value' => $config->get('selected_api'),
      '#description' => $this->t('Select which API to use for processing.'),
      '#required' => TRUE,
    ];
    
    // Add a radio button to select the input method.
    $form['api_input_method'] = [
      '#type' => 'radios',
      '#title' => t('API Key Input Method'),
      '#options' => [
        'api_key' => t('API key in Settings (Testing only)'),
        'file_path' => t('File path (Recommended)'),
      ],
      '#default_value' => $config->get('api_input_method') ? $config->get('api_input_method') : 'api_key',
    ];

    // Add a masked password field for the API key.
    $form['api_key'] = [
      '#type' => 'textfield',
      '#title' => t('API Key'),
      '#description' => t('Enter your API key directly. (Erased when using file based key)'),
      '#default_value' => $config->get('api_key'),
      '#states' => [
        'visible' => [
          ':input[name="api_input_method"]' => ['value' => 'api_key'],
        ],
      ],
      '#attributes' => [
        'id' => 'api-key-field',
      ],
    ];
    
    $recommended_path = $site_root . '/../api_keys/aidmi.key';
    
    $form['api_key_file_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Key File Path'),
      '#description' => $this->t('Provide the full file path to a file that contains your API key. (e.g., outside of web root "/../api_keys/aidmi.key") Actual file location: @path', ['@path' => $recommended_path]),
      '#default_value' => $config->get('api_key_file_path'),
      '#states' => [
        'visible' => [
          ':input[name="api_input_method"]' => ['value' => 'file_path'],
        ],
      ],
    ];

    // Use Drupal's built-in status message system for alerts.
    $form['status_messages'] = [
      '#type' => 'status_messages',
      '#attributes' => ['role' => 'alert', 'aria-live' => 'polite'],
      '#states' => [
        'visible' => [
          ':input[name="api_input_method"]' => ['value' => 'file_path'],
        ],
      ],
    ];

    // Field for the API instructions
    $form['api_instructions'] = [
      '#type' => 'textarea',
      '#title' => $this->t('API Instructions'),
      '#default_value' => $config->get('api_instructions'),
      '#description' => $this->t('Provide the string of instructions to be passed to the API.'),
      '#required' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {

        // Prepare the config object for saving.
        $config = $this->configFactory()->getEditable('aidmi.settings');

        // Get the selected API input method.
        $api_input_method = $config->get('api_input_method');

        if ($api_input_method === 'file_path') {
          // Clear the old API key if the file path is chosen.
          $config->set('api_key', '');
        } else {
          // Save the entered API key if the API key method is chosen.
          $config->set('api_key', $form_state->getValue('api_key'));
        }
    
        // Save the API input method and the file path (if applicable).
        $config->set('api_input_method', $api_input_method)
          ->set('api_input_method', $form_state->getValue('api_input_method'))  // Save selected API Key method      
          ->set('selected_api', $form_state->getValue('selected_api'))  // Save selected API
          ->set('api_key_file_path', $form_state->getValue('api_key_file_path'))
          ->set('api_instructions', $form_state->getValue('api_instructions'))
          ->save();

    parent::submitForm($form, $form_state);
  }
}
