let activeEditorInstance;
let lastSelection;

(function ($, Drupal, once) {
  Drupal.behaviors.ckeditorImageUploadMonitor = {
    attach: function (context, settings) {

      // Capture the selection before opening the dialog.
      function captureSelection(editorInstance) {
        const selection = editorInstance.model.document.selection;
      
        // If there's a selection, clone it to restore later.
        if (selection) {
          lastSelection = selection.getFirstRange().clone();
        }
      }

      // Restore that last selection.
      function restoreSelection(editorInstance) {
        if (lastSelection) {
          editorInstance.model.change(writer => {
            // Restore the selection.
            writer.setSelection(lastSelection);
          });
      
          // Focus the editor view.
          editorInstance.editing.view.focus();
        }
      }

      // Set the last active CKEditor.
      once('getActiveCKEditorInstance', '.ck-editor__editable', context).forEach(function (editorElement) {
        // Attach a focus event listener to detect when an editor becomes active.
        editorElement.addEventListener('focus', function () {
          activeEditorInstance = editorElement.ckeditorInstance;
        });
      });


      // Select the target node (element) you want to observe.
      const targetNode = document.body; // You can change this to any specific element.

      // Define the configuration for the observer (which types of changes to monitor).
      const config = {
        childList: true,       // Monitor additions or removals of child elements.
        attributes: true,      // Monitor changes to attributes.
        characterData: false,  // Don't monitor changes to text content.
        subtree: true,         // Extend monitoring to all descendants of the observed node.
        attributeFilter: ['class', 'src'],  // Only monitor changes to 'class' and 'src' attributes.
        attributeOldValue: true // Record the previous value of changed attributes.
      };

      // Define the callback function that will be triggered when mutations are observed.
      const callback = (mutationsList) => {
        mutationsList.forEach((mutation) => {    
          if (mutation.type === 'attributes') {
            const regex = /^ck-labeled-field-view/;
            if (typeof mutation.target !== 'undefined' && regex.test(mutation.target.id)) {
              altTextField = mutation.target;
              imgTag = activeEditorInstance.data.stringify(activeEditorInstance.model.getSelectedContent(activeEditorInstance.model.document.selection));
              if (altTextField) {
                aidmiButton(altTextField, activeEditorInstance, imgTag);
              }
            }
          }
        });
      };

      // Create an instance of MutationObserver and pass the callback function.
      const observer = new MutationObserver(callback);    
      // Start observing the target node with the provided configuration.
      observer.observe(targetNode, config);

      // Function to add the AIDmi Button.
      function aidmiButton(altTextField, ckEditorInstance, imgTag) {
        // Check if the image has a data-entity-uuid.
        uuidMatch = imgTag.match(/^<img[^>]*data-entity-uuid="([^"]*)"/);
        if (uuidMatch) {
          imgUuid = uuidMatch[1];
        }
        const existingButton = document.querySelector('.aidmi-button');  // Check if the button already exists.
      
        if (!existingButton) {
          const button = document.createElement('button');
          button.type = 'button';  // Set the button type to 'button' to avoid form submission behavior.
          button.textContent = 'AI, describe my image!';
          button.classList.add('aidmi-button');
          // Attach the click event to disable the button after click.
          button.onclick = (e) => {
            e.preventDefault();
            // Disable the button immediately after click.
            button.disabled = true;
            button.textContent = 'AIDmi Processing...';
            // Call the aidmiAjax function and handle the result with .then().
            imgDesc = aidmiAjax(imgUuid)
                        .then((data) => {
                          // Open the Off-Canvas Dialog and show the retrieved description.
                          openOffCanvasDialog(data, (isConfirmed) => {
                            if (isConfirmed) {
                              // Check if the imgTag has an alt attribute.
                              updateImageAltInCKEditor(ckEditorInstance, imgUuid, data);
                            }
                          });
                        })
                        .catch((error) => {
                          console.log('Error:', error);  // Handle errors here.
                          alert(Drupal.t('AI connection error, please contact support.'));
                        })
                        .finally(() => {
                          // Re-enable the button once the AJAX call is complete.
                          button.disabled = false;  // Re-enable the button.
                          button.textContent = 'AI, describe my image!';
                        });
            };
            // Add the button to the DOM.
            altTextField.parentElement.appendChild(button);
        }
      }

      function aidmiAjax(uuid) {
        // Initially clear the field or show a placeholder/loading message.
        altTextField.value = '';  // Keep the field empty.
        // Return a Promise that resolves when the AJAX call succeeds.
        return new Promise((resolve, reject) => {
          $.ajax({
            url: '/admin/aidmi/describe-image-ajax/' + uuid,
            type: 'GET',
            success: function (data) {
              // Resolve the promise with the data from the AJAX call.
              resolve(data);
            },
            error: function (xhr, status, error) {
              // Reject the promise if there's an error.
              reject(error);
            }
          });
        });
      }

      // Open dialog for AIDmi.
      function openOffCanvasDialog(content, callback) {
        // Capture the current selection before opening the dialog
        captureSelection(activeEditorInstance);
        // Create a temporary container to hold the content.
        const tempElement = document.createElement('div');
        tempElement.setAttribute('role', 'dialog');
        tempElement.setAttribute('aria-labelledby', 'aidmi-dialog-title');
        tempElement.setAttribute('aria-describedby', 'aidmi-dialog-description');

        // Set title text.
        let tempETitle = 'AIDmi Description';
        let tempESubTitle = 'Please evaluate for relevance and accuracy.';
        
        // Create screen for dialog.
        tempElement.innerHTML = `
          <div>
            <p id="aidmi-dialog-subtitle" class="aidmi-dialog-subtitle">${Drupal.t(tempESubTitle)}</p>
            <div id="aidmi-dialog-description" class="aidmi-dialog-content-scrollable aidmi-dialog-description">${Drupal.t(content)}</div>
          </div>`;

        // Store the element that triggered the dialog, to return focus later.
        const previousFocus = document.activeElement;
        
        // Use the Drupal off-canvas dialog to show the content.
        const options = {
          dialogClass: 'aidmi-dialog',
          title: tempETitle,
          width: '400px',
          buttons: [
            {
              text: Drupal.t('Insert Text'),
              click: function() {
                // Return true when Insert Text is clicked.
                callback(true);
                // Close the dialog.
                $(tempElement).dialog('close');
              }
            },
            {
              text: Drupal.t('Cancel'),
              click: function() {
                // Return false when Cancel is clicked.
                callback(false);
                // Close the dialog.
                $(tempElement).dialog('close');
              }
            }
          ],
          close: function () {
            // Return focus to the original element when the dialog is closed.
            previousFocus.focus();
      
            // Restore the CKEditor selection after closing the dialog
            restoreSelection(activeEditorInstance);
          }
        };  
            
        // Open the dialog using Drupal's dialog API.
        const dialogInstance = Drupal.dialog(tempElement, options);
        dialogInstance.showModal();

        // Set focus on the description to be read by screen readers.
        const descriptionElement = document.getElementById('aidmi-dialog-description');
        descriptionElement.setAttribute('tabindex', '-1');
        // Ensure Esc key closes the dialog
        document.addEventListener('keydown', function(event) {
          if (event.key === 'Escape') {
            $(tempElement).dialog('close');
          }
        });
      }

      // Update image alt tag in CKEditor
      function updateImageAltInCKEditor(editorInstance, imgUuid, newAltText) {
        // Get the model from the CKEditor instance.
        const model = editorInstance.model;
      
        // Use the model.change() to make sure CKEditor is aware of the modification.
        model.change(writer => {
          // Get the root of the document.
          const root = model.document.getRoot();
      
          // Traverse the root to find the image element by uuid.
          for (const item of model.createRangeIn(root)) {
            const element = item.item;
      
            // Check if the element is an image and matches the imgUuid
            if (element.is('element', 'imageBlock') || element.is('element', 'imageInline')) {
              // Get the element in CKEditor by the UUID.
              const uuid = element.getAttribute('dataEntityUuid');
              if (uuid === imgUuid) {
                // Update the alt attribute using CKEditor's writer
                writer.setAttribute('alt', Drupal.t(newAltText), element);
                // Set the selection on the image element.
                const range = model.createRangeOn(element);
                writer.setSelection(range);
              }
            }
          }
        });
      }
    }
  };
})(jQuery, Drupal, once);
