<?php

namespace Drupal\aidmi\Service;

use GeminiAPI\Client;
use GeminiAPI\Enums\MimeType;
use GeminiAPI\Resources\Parts\ImagePart;
use GeminiAPI\Resources\Parts\TextPart;
use Drupal\file\Entity\File;  // Correct namespace for Drupal File Entity
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Component\Utility\Html;

class GeminiAiService {

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The API client for Google Gemini.
   *
   * @var \GeminiAPI\Client
   */
  protected $client;

  /**
   * The API instructions.
   *
   * @var string
   */
  protected $apiInstructions;

  public function __construct(ConfigFactoryInterface $configFactory) {
    $this->configFactory = $configFactory;
    
    // Retrieve the configuration for the module
    $config = $configFactory->get('aidmi.settings');

    // Initialize the API key variable.
    $apiKey = '';

    // Check if the API key is to be retrieved from the file.
    if ($apiInputMethod === 'file_path') {
      $siteRoot = \Drupal::root(); // Get the site root.
      $fullFilePath = $siteRoot . '/' . ltrim($apiKeyFilePath, '/'); // Full file path.
      // Check if the file exists and is readable.
      if (file_exists($fullFilePath) && is_readable($fullFilePath)) {
        // Read the API key from the file.
        $apiKey = trim(file_get_contents($fullFilePath));
      } else {
        // Log an error and default the API key to an empty string or a default value.
        \Drupal::logger('aidmi')->error('API key file not found or not readable at @path', ['@path' => $fullFilePath]);
        $apiKey = '0'; // Fallback if file is not found or readable.
      }
    } else {
      // Get the API key from the configuration if the input method is 'api_key'.
      $apiKey = $config->get('api_key');
      
      // Default to '0' if the API key is empty.
      if (empty($apiKey)) {
        $apiKey = '0';
      }
    }
    
    $this->client = new Client($apiKey);
    $this->apiInstructions = $config->get('api_instructions');
  }

  public function analyzeImage(int $fid): string {
    try {
      $file = File::load($fid);
      if (!$file) {
        throw new \Exception($this->t('File not found.'));
      }
  
      $imagePath = $file->getFileUri();
      $imageContent = file_get_contents($imagePath);
      if (!$imageContent) {
        throw new \Exception($this->t('Unable to open image file.'));
      }
  
      $response = $this->client->geminiProFlash1_5()->generateContent(
          new TextPart($this->apiInstructions),
          new ImagePart(
              MimeType::IMAGE_JPEG,
              base64_encode($imageContent)
          )
      );
  
      // Return the description from Gemini API
      $output = addslashes($response->text());
      return $output;
    } catch (\Exception $e) {
        return 'Error: ' . $e->getMessage();
    }
  }
  
}
