<?php

namespace Drupal\aidmi\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\file\Entity\File;
// Correct namespace for Drupal File Entity.
use Drupal\file\FileInterface;

/**
 *
 */
class GeminiAiService {
  /**
   * Gemini Base URL and Header for API Key.
   */
  private string $baseUrl = 'https://generativelanguage.googleapis.com';
  private string $apiKeyHeader = 'x-goog-api-key';

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The API instructions.
   *
   * @var string
   */
  protected $apiInstructions;

  /**
   * The API Key.
   *
   * @var string
   */
  protected $apiKey;

  /**
   * Gemini Model.
   *
   * @var string
   */
  protected $geminiModel;

  /**
   * The AIDmi controller.
   *
   * @var \Drupal\aidmi\Controller\AidmiController
   */
  protected $aidmiController;

  public function __construct(ConfigFactoryInterface $configFactory) {
    $this->configFactory = $configFactory;
    // Retrieve the configuration for the module.
    $config = $configFactory->get('aidmi.settings');
    // Get the input method.
    $apiInputMethod = $config->get('api_input_method');

    // Initialize the API key variable.
    $apiKey = '';

    // Check if the API key is to be retrieved from the file.
    if ($apiInputMethod === 'file_path') {
      // Get the API key file path from the configuration.
      $apiKeyFilePath = $config->get('api_key_file_path');
      // Get the site root.
      $siteRoot = \Drupal::root();
      // Full file path.
      $fullFilePath = $siteRoot . '/' . ltrim($apiKeyFilePath, '/');
      // Check if the file exists and is readable.
      if (file_exists($fullFilePath) && is_readable($fullFilePath)) {
        // Read the API key from the file.
        $apiKey = trim(file_get_contents($fullFilePath));
      }
      else {
        // Log an error and default the API key.
        \Drupal::logger('aidmi')->error('API key file not found or not readable at @path', ['@path' => $fullFilePath]);
        // Fallback if file is not found or readable.
        $apiKey = '0';
      }
    }
    else {
      // Get the API key from the configuration.
      $apiKey = $config->get('api_key');

      // Default to '0' if the API key is empty.
      if (empty($apiKey)) {
        $apiKey = '0';
      }
    }

    $this->apiKey = $apiKey;
    $this->geminiModel = $config->get('api_gemini_model');
    $this->apiInstructions = $config->get('api_instructions');
  }

  /**
   * Analyze an individual image.
   *
   * @param string $fid
   *   The fid of the file
   *
   * @return string
   *   Output from Gemini API.
   */
  public function analyzeImage(int $fid): string {
    try {
      $file = File::load($fid);
      if (!$file) {
        throw new \Exception(t('File not found.'));
      }

      $imagePath = $file->getFileUri();
      $imageContent = file_get_contents($imagePath);
      if (!$imageContent) {
        throw new \Exception(t('Unable to open image file.'));
      }

      $output = $this->geminiReviewImage($imagePath);

      return $output;
    }
    catch (\Exception $e) {
      return 'Error: ' . $e->getMessage();
    }
  }

  /**
   * Analyze HTML content and attach any images.
   *
   * @param string $content
   *   HTML content.
   *
   * @param string $imagesJSON
   *   The list of images in JSON form.
   *
   * @return string
   *   Output from Gemini API.
   */
  public function analyzeContent(string $content, string $imagesJSON): string {
    try {
      $output = $this->geminiReviewContent($content, $imagesJSON);
      $output = str_replace("```json", "", $output);
      $output = str_replace("```", "", $output);

      return $output;
    }
    catch (\Exception $e) {
      return 'Error: ' . $e->getMessage();
    }
  }

  /**
   * Get a file based on UUID.
   *
   * @param string $uuid
   *   UUID of image.
   *
   * @param string $type
   *   If standard image file or media.
   *
   * @return object|null
   *   Output file.
   */
  public function getFileIdByUuid(string $uuid, string $type) {
    $file = NULL;

    // If type is drupal-media, load media entity first.
    if ($type == 'drupal-media') {
      $media = $this->loadMediaByUuid($uuid);
      if ($media && $media->bundle() === 'image') {
        // Check if the media entity has the field.
        if ($media->hasField('field_media_image') && !$media->get('field_media_image')->isEmpty()) {
          $file_id = $media->get('field_media_image')->target_id;
          // Load the file entity using the file ID.
          $file = File::load($file_id);
        }
      }
    }
    else {
      // If type is not drupal-media, directly get file entity by UUID.
      $files = \Drupal::entityTypeManager()->getStorage('file')->loadByProperties(['uuid' => $uuid]);
      if (!empty($files)) {
        $file = reset($files);
      }
    }

    // Check if the file is loaded and valid.
    if ($file instanceof FileInterface) {
      return $file;
    }
    elseif ($media && $media->bundle() != 'image') {
      return NULL;
    }
    else {
      throw new \Exception(t('File not found for UUID: @uuid', ['@uuid' => $uuid]));
    }
  }

  /**
   * Load a media entity by its UUID.
   *
   * @param string $uuid
   *   The UUID of the media entity.
   *
   * @return \Drupal\media\MediaInterface|null
   *   The media entity object if found, otherwise null.
   */
  public function loadMediaByUuid($uuid) {
    // Load the media entity using the UUID.
    $media_entities = \Drupal::entityTypeManager()
      ->getStorage('media')
      ->loadByProperties(['uuid' => $uuid]);

    // Check if any media entities are returned.
    if (!empty($media_entities)) {
      // Get the first media entity from the array.
      return reset($media_entities);
    }

    return NULL;
  }

  /**
   * Ready and send a single image to Gemini.
   *
   * @param string $imagePath
   *   Path to the image.
   *
   * @return string
   *   Output from Gemini API.
   */
  public function geminiReviewImage(string $imagePath):string {
    // Setup request to send json via POST.
    // The text prompt you want to send.
    $prompt = $this->apiInstructions;
    // Read the image file and encode it in base64.
    $imageData = base64_encode(file_get_contents($imagePath));
    // Prepare the data payload.
    $data = [
      'contents' => [
        'parts' => [
          [
            'text' => $prompt,
          ],
          [
            'inlineData' => [
              'mimeType' => 'image/jpeg',
              'data' => $imageData,
            ],
          ],
        ],
      ],
    ];
    
    // Send to Gemini.
    $fullDescription = $this->geminiPost(json_encode($data));

    // Print the analysis result.
    if (isset($fullDescription)) {
      return $fullDescription;
    }
    else {
      return '**Error**';
    }
  }

  /**
   * Ready and send full content and images to Gemini.
   *
   * @param string $content
   *   Full html content from page.
   *
   * @param string $imagesJSON
   *   JSON list of images from AIDmi.
   *
   * @return string
   *   Output from Gemini API.
   */
  public function geminiReviewContent(string $content, string $imagesJSON):string {
    // Decode JSON string into PHP array.
    $jsonArray = json_decode($imagesJSON, TRUE);

    // Setup request to send json via POST.
    // The text prompt you want to send.
    $prompt = $this->apiInstructions;
    $prompt = 'Response can only be in JSON. Do not provide null value if something is empty, just keep it "" blank. Exact JSON output example: [{"images": [{"data-entity-uuid": "", "src": "", "before_alt": "", "recommendation": "", "decorative_image" = ""}]}, {"message": "error or other message if needed"}]. Review this content and describe the image. Actual images from the content are attached and referenced by data-entity-uuid. Provide a json output with each image by data-entity-uuid, image src, the before alt text if available, and your recommendation. If there is a problem, respond with only an error JSON. Provide a true or false for decorative image where just an alt tag alone is decorative, but alt="" is false. ' . $prompt;

    $generateContent = [];

    $generateContent[] = ['text' => $prompt];

    foreach ($jsonArray as $item) {
      $imagePath = NULL;
      $imageContent = NULL;

      // Get file entity by UUID.
      $file = $this->getFileIdByUuid($item['data-entity-uuid'], $item['type']);

      if ($file instanceof FileInterface) {
        $imagePath = \Drupal::service('file_system')->realpath($file->getFileUri());
        // Use the file_url_generator service to generate the URL.
        $url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());

        // Check if file path is valid.
        if (file_exists($imagePath)) {
          $imageContent = file_get_contents($imagePath);
        }
      }

      // If there is an image, do it.
      if ($imageContent) {
        // Attach images for Gemini AI review.
        $generateContent[] = ['text' => "data-entity-uuid: " . $item['data-entity-uuid'] . ", src: " . $url];
        $generateContent[] = [
          'inlineData' => [
            'mimeType' => 'image/jpeg',
            'data' => base64_encode($imageContent),
          ],
        ];
      }
    }

    // Read the image file and encode it in base64.
    $imageData = base64_encode(file_get_contents($imagePath));
    // Prepare the data payload.
    $data = [
      'contents' => [
        'parts' => $generateContent,
      ],
    ];

    // Send to Gemini.
    $fullDescription = $this->geminiPost(json_encode($data));

    // Print the analysis result.
    if (isset($fullDescription)) {
      return $fullDescription;
    }
    else {
      return '**Error**';
    }
  }

  /**
   * Post to Gemini API.
   *
   * @param string $data
   *   JSON for Gemini API post.
   *
   * @return string
   *   Output from Gemini API.
   */
  private function geminiPost(string $data):string {    
    // Connect with URL and selected model.
    $fullUrl = "{$this->baseUrl}/v1/{$this->geminiModel}:streamGenerateContent";
    $ch = curl_init($fullUrl);
    
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [$this->apiKeyHeader . ': ' . $this->apiKey, 'Content-Type:application/json']);
    // Return response instead of printing.
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, TRUE);
    // Send request.
    $result = curl_exec($ch);
    curl_close($ch);
    // Decode and handle the response.
    $resultData = json_decode($result, TRUE);
    // Return $result;.
    $fullDescription = NULL;

    // Iterate through each candidate in the response data.
    foreach ($resultData as $entry) {
      if (isset($entry['candidates'][0]['content']['parts'])) {
        // Concatenate all the text parts together.
        foreach ($entry['candidates'][0]['content']['parts'] as $part) {
          if (isset($part['text'])) {
            $fullDescription .= $part['text'];
          }
        }
      }
    }

    // Send results back.
    return $fullDescription;
  }

}
