<?php

namespace Drupal\aidmi\Controller;

use Drupal\aidmi\Service\GeminiAiService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * AIDmi Controller.
 */
class AidmiController {
  protected $geminiAiService;

  public function __construct(GeminiAiService $geminiAiService) {
    $this->geminiAiService = $geminiAiService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('aidmi.ai_service')
    );
  }

  /**
   * Analyze a single image from Ajax.
   *
   * @param string $uuid
   *   UUID of image.
   *
   * @return Response
   *   Output from Gemini API process.
   */
  public function analyzeImageAjax(string $uuid): Response {
    // Get the file by uuid.
    $fid = $this->getFileIdByUuid($uuid);

    // Send it to Gemini API.
    if ($fid) {
      $description = $this->geminiAiService->analyzeImage($fid);
      return new Response($description);
    }
    
    // Return the output.
    return NULL;
  }

  /**
   * Analyze HTML content from Ajax.
   *
   * @return Response
   *   Output from Gemini API process.
   */
  public function analyzeContentAjax(): Response {
    // Get the raw POST data from the request.
    $content = \Drupal::request()->request->get('content');
    $imagesJSON = \Drupal::request()->request->get('imagesJSON');
    $description = NULL;

    // Send it to Gemini API.
    if (!empty($content)) {
      $description = $this->geminiAiService->analyzeContent($content, $imagesJSON);
    }

    // Return the output.
    return new Response($description);
  }

  /**
   * Get file by the UUID
   *
   * @param string $uuid
   *   UUID of image.
   *
   * @return object
   *   Output file.
   */
  public function getFileIdByUuid(string $uuid) {
    // Query for the file entity using the UUID.
    $file = \Drupal::entityTypeManager()->getStorage('file')->loadByProperties(['uuid' => $uuid]);

    // Return file or nothing.
    if ($file) {
      // Get the first result.
      $file = reset($file);
      return $file->id();
    }
    else {
      return NULL;
    }
  }

}
