<?php

namespace Drupal\ajax_big_pipe\Plugin\rest\resource;

use Drupal\ajax_big_pipe\Render\Placeholder\AjaxBigPipeStrategy;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\RemoveCommand;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\PathProcessor\PathProcessorManager;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\path_alias\AliasManager;
use Drupal\rest\Attribute\RestResource;
use Drupal\rest\ModifiedResourceResponse;
use Drupal\rest\Plugin\ResourceBase;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides a resource to get view modes by entity and bundle.
 */
#[RestResource(
  id: 'load_ajax_big_pipe',
  label: new TranslatableMarkup('AJAX BigPipe'),
  uri_paths: [
    'canonical' => '/api/bigpipe',
  ]
)]
class LoadAjaxBigPipe extends ResourceBase {

  /**
   * The path alias manager.
   *
   * @var \Drupal\path_alias\AliasManagerInterface
   */
  protected $pathAliasManager;

  /**
   * The path processor manager.
   *
   * @var \Drupal\Core\PathProcessor\PathProcessorManager
   */
  protected $pathProcessorManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * Constructs a LoadAjaxBigPipe object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param array $serializer_formats
   *   The serializer formats supported by the resource.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack service.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\path_alias\AliasManager $path_alias_manager
   *   The path alias manager service.
   * @param \Drupal\Core\PathProcessor\PathProcessorManager $path_processor_manager
   *   The path processor manager service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   Route parameters for processing requests and handling redirects.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    array $serializer_formats,
    LoggerInterface $logger,
    RequestStack $request_stack,
    RendererInterface $renderer,
    AliasManager $path_alias_manager,
    PathProcessorManager $path_processor_manager,
    EntityTypeManagerInterface $entity_type_manager,
    RouteMatchInterface $route_match,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $serializer_formats, $logger);
    $this->requestStack = $request_stack;
    $this->renderer = $renderer;
    $this->pathAliasManager = $path_alias_manager;
    $this->pathProcessorManager = $path_processor_manager;
    $this->entityTypeManager = $entity_type_manager;
    $this->routeMatch = $route_match;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->getParameter('serializer.formats'),
      $container->get('logger.factory')->get('ajax_big_pipe'),
      $container->get('request_stack'),
      $container->get('renderer'),
      $container->get('path_alias.manager'),
      $container->get('path_processor_manager'),
      $container->get('entity_type.manager'),
      $container->get('current_route_match')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getBaseRouteRequirements($method) {
    return [
      '_access' => 'TRUE',
    ];
  }

  /**
   * Responds to GET requests.
   */
  public function get() {

    if ($this->requestStack->getCurrentRequest()->cookies->has(AjaxBigPipeStrategy::NOJS_COOKIE)) {
      return new ModifiedResourceResponse([], 200);
    }
    $response = new AjaxResponse();

    $request = $this->requestStack->getCurrentRequest()->query->all();

    if (!empty($request['ajax_page_state'])) {
      $this->requestStack->getCurrentRequest()->request->set('ajax_page_state', $request['ajax_page_state']);
    }

    if (!empty($request['destination'])) {
      $destination = parse_url($request['destination']);
      if (!empty($destination['path'])) {

        $base_path = $this->requestStack->getCurrentRequest()->getBasePath();
        if (!empty($base_path)) {
          $destination['path'] = str_replace($base_path, '', $destination['path']);
        }

        $result = $this->pathAliasManager->getPathByAlias($destination['path']);

        $path = $this->pathProcessorManager->processInbound($result, $this->requestStack->getCurrentRequest());
        if (!empty($path)) {
          $result = $path;
        }

        $url = Url::fromUri("internal:" . $result);
        if ($url->isRouted()) {
          $params = $url->getRouteParameters();
          if (!empty($params)) {
            foreach ($params as $key => $param) {
              switch ($key) {
                case 'taxonomy_term':
                case 'filter_entity':
                case 'node':
                case 'user':
                  $this->routeMatch->getParameters()->set($key, $this->entityTypeManager->getStorage($key)->load($param));
                  break;

                default:
                  $this->routeMatch->getParameters()->set($key, $param);
                  break;
              }
            }
          }
        }
      }
    }

    $callback = $request['callback'] ?? '';
    $args = $request['args'] ?? [];
    $token = $request['token'] ?? '';
    if ($this->validateToken($callback, $args, $token) && is_callable($callback)) {
      $render_array = [
        '#lazy_builder' => [$callback, $args],
        '#create_placeholder' => FALSE,
      ];
      $html = $this->renderer->renderRoot($render_array);

      $response->setAttachments($render_array['#attached']);

      if (!empty($html)) {
        $response->addCommand(new ReplaceCommand(NULL, $html));
      }
      else {
        $response->addCommand(new RemoveCommand(NULL));
      }
    }

    return $response;
  }

  /**
   * Validates the provided token against a generated token.
   *
   * @param string $callback
   *   The callback function name used to generate the token.
   * @param array $args
   *   The arguments passed to the callback for token generation.
   * @param string $provided_token
   *   The token provided for validation.
   */
  private function validateToken(string $callback, array $args, string $provided_token): bool {
    return AjaxBigPipeStrategy::generateToken($callback, $args) == $provided_token;
  }

}
