<?php

namespace Drupal\ajax_dashboard\Form;

use Drupal\ajax_dashboard\AJAXDashboardManagerInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;

class AJAXDashboard extends FormBase {

  /**
   * @inheritdoc
   */
  public function getFormId() {
    return 'ajax_dashboard';
  }

  /**
   * @inheritdoc
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // This form doesn't really submit anything.
  }

  /**
   * This is a debug function, which your plugin can call to test AJAX.
   * @param array $form
   * @param FormStateInterface $form_state
   * @param null $request
   * @return AjaxResponse
   */
  public static function rebuild(array $form, FormStateInterface $form_state, $request = NULL) {
    \Drupal::messenger()->addMessage(0);
    $add = [
      '#type' => 'html_tag',
      '#tag' => 'h2',
      $add['#value'] = t('AJAX Rebuild successful. Please implement a rebuild function in your dashboard plugin.'),
      '#prefix' => '<div id="edit-dashboard" class="dashboard-wrapper"><pre>',
      '#suffix' => '</pre></div>',
    ];
    $response = new AjaxResponse();
    $response->addCommand(new ReplaceCommand('#edit-dashboard', $add));
    return $response;
  }

  /**
   * @param array $form
   * @param FormStateInterface $form_state
   * @param $dashboard_id
   * @param array $context
   * @return array
   */
  public function buildForm(array $form, FormStateInterface $form_state, $dashboard_id = '', array $context = []) {
    $dashboard_manager = \Drupal::service('plugin.manager.ajax_dashboard');
    $dashboard_section_manager = \Drupal::service('plugin.manager.ajax_dashboard_section');
    /** @var $manager AJAXDashboardManagerInterface */
    $dashboards = $dashboard_manager->getDefinitions();
    if (isset($dashboards[$dashboard_id])) {
      $dboard = $dashboards[$dashboard_id];
      $context['dashboard_id'] = $dashboard_id;

      $form['label'] = [
          '#type' => 'html_tag',
          '#tag' => isset($dboard['label_tag']) ? $dboard['label_tag'] : 'h2',
          '#value' => $dboard['label']
        ];
      $form['dashboard_container'] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['ajax-dashboard-container', $dboard['id']],
        ],
      ];
      $form['dashboard_container']['sections'] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['ajax-dashboard-sections'],
        ],
      ];
      $form['dashboard_container']['dashboard'] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['ajax-dashboard-dashboard'],
        ],
      ];

      // Add render attributes to the overall board.
      $render_items = array_filter($dboard, function($k) {return strpos($k, '#') !== FALSE;}, ARRAY_FILTER_USE_KEY);
      foreach ($render_items as $key => $value) {
        $form['dashboard_container'][$key] = $value;
      }

      // Sections Container
      if (isset($dboard['sections'])) {
        $render_items = array_filter($dboard['sections'], function($k) {return strpos($k, '#') !== FALSE;}, ARRAY_FILTER_USE_KEY);
        $sections = array_filter($dboard['sections'], function($k) {return strpos($k, '#') === FALSE;}, ARRAY_FILTER_USE_KEY);
        // Sort the dashboard items.
        $sort_func = function($a, $b) {
          $wa = isset($a['weight']) ? $a['weight'] : 0;
          $wb = isset($b['weight']) ? $b['weight'] : 0;
          if ($wa == $wb) {
            return 0;
          }
          return ($wa < $wb) ? -1 : 1;
        };
        uasort($sections, $sort_func);

        foreach ($render_items as $key => $value) {
          $form['dashboard_container']['sections'][$key] = $value;
        }
        // Get the plugins for the section rendering.
        $plugins = $dashboard_section_manager->getDefinitions();
        $i = 0;
        foreach ($sections as $key => $section) {
          // Check that we have a render class.
          if (isset($plugins[$section['plugin']]['class'])) {
            $context['position'] = $i;
            // Check for access, then write the dashboard section.
            if ($plugins[$section['plugin']]['class']::access($context, $section)) {
              // Write the dashboard.
              $write = FALSE;
              // Disqualify $i = 0 case.
              $disqualify = FALSE;
              // Check for a cookie match.
              $cookies = \Drupal::request()->cookies;
              if ($cookies->has('ajax_dashboard__' . $dashboard_id)) {
                $write = $cookies->get('ajax_dashboard__' . $dashboard_id) === $key;
                $disqualify = !$write;
              }
              // Query match overrides cookie match.
              $query = \Drupal::request()->query;
              if ($query->has('dashboard')) {
                $write = $query->get('dashboard') === $key;
                $disqualify = !$write;
              }

              if ($write || (!$disqualify && $i === 0)) {
                // This catches cases in an AJAX reload where an AJAXResponse is returned instead of a render array.
                $content = $plugins[$section['plugin']]['class']::buildDashboard($form, $form_state, $context, $section);
                if(is_array($content)) {
                  $form['dashboard_container']['dashboard']['content'] = $content;
                }
              }

              $i++;
              $form['dashboard_container']['sections'][$key] = $plugins[$section['plugin']]['class']::buildSection($form, $form_state, $context, $section);
              $form['dashboard_container']['sections'][$key]['#attributes']['data-ajax-dashboard-key'] = $key;
              if (isset($form['dashboard_container']['sections'][$key]['#attributes']['class'])) {
                $form['dashboard_container']['sections'][$key]['#attributes']['class'][] = 'ajax-dashboard-button';
              }
              else {
                $form['dashboard_container']['sections'][$key]['#attributes']['class'] = ['ajax-dashboard-button'];
              }
            }
          }
        }
      }
      // Dashboard render attributes, on initial build
      if (isset($dboard['dashboard']) && !$form_state->getTriggeringElement()) {
        $render_items = array_filter($dboard['dashboard'], function($k) {return strpos($k, '#') !== FALSE;}, ARRAY_FILTER_USE_KEY);
        foreach ($render_items as $key => $value) {
          if ($key === '#attributes' && is_array($value)) {
            foreach ($value as $k => $v) {
              if (is_array($v)) {
                $form['dashboard_container']['dashboard']['#attributes'][$k] = isset($form['dashboard']['#attributes'][$k]) ? array_unique(array_merge($form['dashboard']['#attributes'][$k], $v)) : $v;
              }
              else {
                $form['dashboard_container']['dashboard']['#attributes'][$k] = $v;
              }
            }
          }
          else {
            $form['dashboard_container']['dashboard'][$key] = $value;
          }
        }
      }
    }
    $form['#cache']['contexts'] = ['user', 'url'];
    $form['#attached']['library'][] = 'ajax_dashboard/dashboard';
    $form['#attached']['drupalSettings']['ajax_dashboard']['dashboard_id'] = $dashboard_id;
    return $form;
  }

}
