/**
 * @file
 * Provides Ajaxin functionality.
 */

/* global window, document, define, module */
(function (root, factory) {

  'use strict';

  // Inspired by https://github.com/addyosmani/memoize.js/blob/master/memoize.js
  if (typeof define === 'function' && define.amd) {
    // AMD. Register as an anonymous module.
    define([window.Drupal, window.drupalSettings, window.dBlazy], factory);
  }
  else if (typeof exports === 'object') {
    // Node. Does not work with strict CommonJS, but only CommonJS-like
    // environments that support module.exports, like Node.
    module.exports = factory(window.Drupal, window.drupalSettings, window.dBlazy);
  }
  else {
    // Browser globals (root is window).
    root.Ajaxin = factory(window.Drupal, window.drupalSettings, window.dBlazy);
  }
})(this, function (Drupal, drupalSettings, dBlazy) {

  'use strict';

  /**
   * Theme function for a standalone loading animation.
   *
   * @param {Object} settings
   *   An object containing:
   *   - theme: The inline theme.
   *   - themeFullscreen: The fullscreen theme.
   *   - fs: Whether to use the fullscreen wrapper.
   *
   * @return {HTMLElement}
   *   Returns a HTMLElement object.
   */
  Drupal.theme.ajaxin = function (settings) {
    var config = dBlazy.extend({}, drupalSettings.ajaxin || {}, settings || {});
    var fs = config.fs || false;
    return fs ? config.themeFullscreen : config.theme;
  };

  /**
   * Constructor for Ajaxin.
   *
   * @param {object} options
   *   The Ajaxin options.
   *
   * @namespace
   */
  function Ajaxin(options) {
    var me = this;

    me.options = dBlazy.extend({}, me.defaults, options || {});

    init(me);
  }

  // Cache our prototype.
  var _proto = Ajaxin.prototype;
  _proto.constructor = Ajaxin;

  _proto.defaults = {
    root: null,
    successClass: 'b-loaded',
    selector: '.b-lazy',
    errorClass: 'b-error',
    loaderClass: 'b-loader',
    oldLoadingClass: false,
    doneListener: false,
    inside: false,
    addNow: false
  };

  _proto.isLoaded = function (el) {
    return this.has(el) && el.classList.contains(this.options.successClass);
  };

  _proto.sibling = function (el) {
    return el.nextElementSibling !== null && el.nextElementSibling.classList.contains(this.options.loaderClass) ? el.nextElementSibling : null;
  };

  _proto.loader = function (el) {
    return this.options.inside ? el.querySelector('.' + this.options.loaderClass) : this.sibling(el);
  };

  _proto.has = function (el) {
    return this.loader(el) !== null;
  };

  _proto.add = function (el) {
    var me = this;
    var settings = drupalSettings.ajaxin || {};
    settings.fs = false;
    if (el !== null && !me.has(el)) {
      var template = Drupal.theme('ajaxin', settings);
      el.insertAdjacentHTML(me.options.inside ? 'beforeend' : 'afterend', template);
    }
  };

  _proto.remove = function (el) {
    var me = this;

    // Don't bother removing as when the loader is placed inside, it will be
    // removed automatically on content replacement via .html() alike method.
    if (me.options.inside) {
      return;
    }

    // DOM ready fix.
    window.setTimeout(function () {
      if (me.isLoaded(el) && el.parentNode !== null) {
        el.parentNode.removeChild(el.nextElementSibling);
      }
    }, 100);
  };

  _proto.onDone = function (e) {
    var me = this;
    var el = e.target;

    me.remove(el);
    if (me.options.doneListener) {
      me.unbindEvent(el, me.options.doneListener, me.onDone);
    }

    me.count--;

    if (me.count === 0 && !me.destroyed) {
      me.destroy();
      me.destroyed = true;
    }
  };

  _proto.unbindEvent = function (el, type, fn) {
    el.removeEventListener(type, fn, {capture: false, passive: true});
  };

  _proto.bindEvents = function () {
    var me = this;
    var elms = document.querySelectorAll('.ajaxin-wrapper');

    var doClick = function (e) {
      var el = e.target;
      // Simply remove the loading animation on clicking it.
      if (el.parentNode !== null) {
        el.parentNode.removeChild(el);
      }

      me.unbindEvent(el, 'click', doClick);
    };

    dBlazy.forEach(elms, function (el) {
      // Binds event to `click`.
      el.addEventListener('click', doClick, false);
    });
  };

  _proto.buildOut = function () {
    var me = this;

    if (me.elms === null || !me.elms.length) {
      return;
    }

    dBlazy.forEach(me.elms, function (el) {
      if (me.options.oldLoadingClass) {
        var oldEl = dBlazy.closest(el, '.' + me.options.oldLoadingClass);
        if (oldEl !== null) {
          oldEl.classList.remove(me.options.oldLoadingClass);
        }
      }
      // Appends the progressbar.
      if (!me.isLoaded(el)) {
        // Attaches our loading animation either as a sibling, or child element.
        me.add(el);

        // Binds event to `blazy.done`.
        if (me.options.doneListener) {
          el.addEventListener(me.options.doneListener, dBlazy.throttle(me.onDone.bind(me), 250, me), false);
        }
      }
    });
  };

  _proto.destroy = function () {
    var me = this;
    me.elms = null;
    me.count = 0;
    me.destroyed = true;
  };

  _proto.reinit = function () {
    var me = this;

    me.destroyed = false;

    init(me);
  };

  function init(me) {
    me.options.selector = me.options.selector + ':not(.' + me.options.successClass + '):not(iframe)';
    me.elms = (me.options.root || document).querySelectorAll(me.options.selector);
    me.count = me.elms.length;
    me.destroyed = false;

    if (me.options.addNow) {
      me.buildOut();
    }

    me.bindEvents();
  }

  return Ajaxin;

});
