#!/usr/bin/env php
<?php

$project_name = $argv[1] ?? getenv('CI_PROJECT_NAME');
if (empty($project_name)) {
  throw new RuntimeException('Unable to determine project name.');
}

$path = 'composer.json';
$json_project = json_decode(file_get_contents($path), TRUE);
$json_default = default_json($project_name);
// Avoid adding core-recommended if drupal/core is present.
if (isset($json_project['require-dev']['drupal/core'])) {
  unset($json_default['require-dev']['drupal/core-recommended']);
}
// Conditionally add prophesy
if (!isset($json_project['require-dev']['phpspec/prophecy-phpunit']) && version_compare(getenv('_TARGET_CORE'), '9.0.0', '>=')) {
  $json_default['require-dev']['phpspec/prophecy-phpunit'] = '^2';
}
$json_rich = merge_deep($json_default, $json_project);
// Remove empty top-level items.
$json_rich = array_filter($json_rich);
file_put_contents(getenv('COMPOSER') ?? $path, json_encode($json_rich, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_AMP | JSON_HEX_QUOT));

/**
 * Get default composer.json contents.
 */
function default_json(string $project_name): array {
  $drupalConstraint = getenv('DRUPAL_CORE_CONSTRAINT') ?: '^9';
  $webRoot = getenv('WEB_ROOT') ?: 'web';
  $webPort = getenv('WEB_PORT') ?: '8888';
  return [
    'name' => 'drupal/' . $project_name,
    'type' => 'drupal-module',
    'description' => 'A description',
    'license' => 'GPL-2.0-or-later',
    'repositories' => [
      [
        'type' => 'composer',
        'url' => 'https://packages.drupal.org/8',
      ],
    ],
    'require' => [],
    'require-dev' => [
      'composer/installers' => '^1 || ^2',
      'drupal/core-composer-scaffold' => $drupalConstraint,
      'cweagans/composer-patches' => '~1.0',
      'drupal/core-recommended' => $drupalConstraint,
      'drupal/core-dev' => $drupalConstraint,
      'drush/drush' => '^10 || ^11 || ^12',
      'php-parallel-lint/php-parallel-lint' => '^1.2',
    ],
    'minimum-stability' => 'dev',
    'scripts' => [
      'si' => 'drush si -v --db-url=${SIMPLETEST_DB:-mysql://root:password@mariadb/db}',
      'phpcs' => 'phpcs -s ' . $webRoot . '/modules/custom',
      'phpcbf' => 'phpcbf -s ' . $webRoot . '/modules/custom',
      'lint' => 'parallel-lint --exclude ' . $webRoot . ' --exclude vendor .',
      'webserver' => 'cd ' . $webRoot . ' && php -S 0.0.0.0:' . $webPort . ' .ht.router.php',
      'chromedriver' => 'chromedriver --port=9515 --verbose --whitelisted-ips --log-path=/tmp/chromedriver.log --no-sandbox',
      'unit' => 'SIMPLETEST_DB=${SIMPLETEST_DB:-mysql://root:password@mariadb/db} SIMPLETEST_BASE_URL=${SIMPLETEST_BASE_URL:-http://0.0.0.0:' . $webPort . '} vendor/bin/phpunit --printer="\Drupal\Tests\Listeners\HtmlOutputPrinter" --bootstrap $PWD/web/core/tests/bootstrap.php ' . $webRoot . '/modules/custom',
      'stylelint' => 'yarn --silent --cwd ' . $webRoot . '/core stylelint --formatter verbose --config ./.stylelintrc.json ../modules/custom/**/*.css',
      // See prerequisites at https://www.drupal.org/docs/develop/standards/javascript/eslint-settings#s-checking-custom-javascript-with-eslint
      'eslint' => 'yarn --cwd ' . $webRoot . '/core --config .eslintrc.legacy.json eslint ../modules/custom/**/*.js',
    ],
    'config' => [
      'process-timeout' => 36000,
      "allow-plugins" => [
        "dealerdirect/phpcodesniffer-composer-installer" => TRUE,
        "composer/installers" => TRUE,
        "cweagans/composer-patches" => TRUE,
        "drupal/core-composer-scaffold" => TRUE,
        "drupalspoons/composer-plugin" => TRUE,
        "phpstan/extension-installer" => TRUE,
      ],
    ],
    'extra' => [
      'installer-paths' => [
        $webRoot . '/core' => [
          0 => 'type:drupal-core',
        ],
        $webRoot . '/libraries/{$name}' => [
          0 => 'type:drupal-library',
        ],
        $webRoot . '/modules/contrib/{$name}' => [
          0 => 'type:drupal-module',
        ],
        $webRoot . '/profiles/{$name}' => [
          0 => 'type:drupal-profile',
        ],
        $webRoot . '/themes/{$name}' => [
          0 => 'type:drupal-theme',
        ],
        'drush/{$name}' => [
          0 => 'type:drupal-drush',
        ],
      ],
      'drupal-scaffold' => [
        'locations' => [
          'web-root' => $webRoot . '/',
        ],
      ],
      'drush' => [
        'services' => [
          'drush.services.yml' => '^9 || ^10 || ^11',
        ],
      ],
    ],
  ];
}

/**
 * Deeply merges arrays. Borrowed from Drupal core.
 */
function merge_deep(): array {
  return merge_deep_array(func_get_args());
}

/**
 * Deeply merges arrays. Borrowed from drupal.org/project/core.
 *
 * @param array $arrays
 *   An array of array that will be merged.
 * @param bool $preserve_integer_keys
 *   Whether to preserve integer keys.
 */
function merge_deep_array(array $arrays, bool $preserve_integer_keys = FALSE): array {
  $result = [];
  foreach ($arrays as $array) {
    foreach ($array as $key => $value) {
      if (is_int($key) && !$preserve_integer_keys) {
        $result[] = $value;
      }
      elseif (isset($result[$key]) && is_array($result[$key]) && is_array($value)) {
        $result[$key] = merge_deep_array([$result[$key], $value], $preserve_integer_keys);
      }
      else {
        $result[$key] = $value;
      }
    }
  }
  return $result;
}
