<?php

namespace Drupal\akamai\Helper;

use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Provides formatting to help cache tags conform Akamai requirements.
 *
 * @see https://developer.akamai.com/api/purge/ccu/overview.html
 */
class CacheTagFormatter {

  const RESTRICTED_CHARS_PATTERN = '/[\s\*\"\(\)\,\:\;\<\=\>\?\@\\\[\]\{\}]/';

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a CacheTagFormatter helper object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * Format tag according to Akamai guidelines.
   *
   * @param string $tag
   *   A cache tag string.
   *
   * @return string
   *   A compliant cache tag string.
   */
  public function format($tag) {
    /** @var \Drupal\Core\Config\Config $config */
    $config = $this->configFactory->get('akamai.settings');
    $tag = (string) $tag;
    // Add cache tag prefix (for use with multisites)
    $tag = $config->get('edge_cache_tag_header_prefix') . $tag;
    $tag = preg_replace(self::RESTRICTED_CHARS_PATTERN, '_', $tag);
    return $tag;
  }

}
